<?php

/**
 * Class to handle all db operations
 * This class will have CRUD methods for database tables
 *
 * @author Ravi Tamada
 * @link URL Tutorial link
 */
class DbHandler {

    private $conn;

    function __construct() {
        require_once dirname(__FILE__) . '/DbConnect.php';
        // opening db connection
        $db = new DbConnect();
        $this->conn = $db->connect();
    }

    /* ------------- `users` table method ------------------ */

    /**
     * Creating new user
     * @param String $name User full name
     * @param String $email User login email id
     * @param String $password User login password
     */
    public function createUser($name, $email, $password) {
        require_once 'PassHash.php';
        $response = array();

        // First check if user already existed in db
        if (!$this->isUserExists($email)) {
            // Generating password hash
            $password_hash = PassHash::hash($password);

            // Generating API key
            $api_key = $this->generateApiKey();

            // insert query
            $stmt = $this->conn->prepare("INSERT INTO users(name, email, password_hash, api_key, status) values(?, ?, ?, ?, 1)");
            $stmt->bind_param("ssss", $name, $email, $password_hash, $api_key);

            $result = $stmt->execute();

            $stmt->close();

            // Check for successful insertion
            if ($result) {
                // User successfully inserted
                return USER_CREATED_SUCCESSFULLY;
            } else {
                // Failed to create user
                return USER_CREATE_FAILED;
            }
        } else {
            // User with same email already existed in the db
            return USER_ALREADY_EXISTED;
        }

        return $response;
    }

    /** Opal
     * Checking user login
     * @param String $email User login email id
     * @param String $password User login password
     * @return boolean User login status success/fail
     */
    public function checkLogin($email, $password) {
        // fetching user by email
        $stmt = $this->conn->prepare("SELECT password FROM tbl_customer WHERE emailaddress = ? AND status = 1");

        $stmt->bind_param("s", $email);

        $stmt->execute();

        $stmt->bind_result($password_hash);

        $stmt->store_result();

        if ($stmt->num_rows > 0) {
            // Found user with the email
            // Now verify the password

            $stmt->fetch();

            $stmt->close();

            if (PassHash::check_password($password_hash, $password)) {
                // User password is correct
                return TRUE;
            } else {
                // user password is incorrect
                return FALSE;
            }
        } else {
            $stmt->close();

            // user not existed with the email
            return FALSE;
        }
    }

    /** Opal
     * Checking for duplicate user by email address
     * @param String $email email to check in db
     * @return boolean
     */
    private function isUserExists($email) {
        $stmt = $this->conn->prepare("SELECT id from tbl_customer WHERE emailaddress = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $stmt->store_result();
        $num_rows = $stmt->num_rows;
        $stmt->close();
        return $num_rows > 0;
    }

    /** Opal
     * Fetching user by email
     * @param String $email User email id
     */
    public function getUserByEmail($email) {
        $stmt = $this->conn->prepare("SELECT firstname, lastname, emailaddress, api_key, profilepic FROM tbl_customer WHERE emailaddress = ?");
        $stmt->bind_param("s", $email);
		//print_r($stmt);
        if ($stmt->execute()) {
            // $user = $stmt->get_result()->fetch_assoc();
            $stmt->bind_result($firstname, $lastname, $emailaddress, $api_key, $profilepic);
            $stmt->fetch();
            $user = array();
            $user["firstname"] = $firstname;
            $user["lastname"] = $lastname;
            $user["emailaddress"] = $emailaddress;
            $user["api_key"] = $api_key;
			if($profilepic == NULL || $profilepic == ""){
				$user["userImage"] = 'no-profile.jpg';
			}else{
				$user["userImage"] = $profilepic;
			}
			//print_r($user);
            $stmt->close();
            return $user;
        } else {
            return NULL;
        }
    }

    /** Opal
     * Fetching user api key
     * @param String $user_id user id primary key in user table
     */
    public function getApiKeyById($user_id) {
        $stmt = $this->conn->prepare("SELECT api_key FROM tbl_customer WHERE id = ?");
        $stmt->bind_param("i", $user_id);
        if ($stmt->execute()) {
            // $api_key = $stmt->get_result()->fetch_assoc();
            // TODO
            $stmt->bind_result($api_key);
            $stmt->close();
            return $api_key;
        } else {
            return NULL;
        }
    }

    /** Opal
     * Fetching user id by api key
     * @param String $api_key user api key
     */
    public function getUserId($api_key) {
        $stmt = $this->conn->prepare("SELECT user_id FROM tbl_customer WHERE api_key = ?");
        $stmt->bind_param("s", $api_key);
        if ($stmt->execute()) {
            $stmt->bind_result($user_id);
            $stmt->fetch();
            // TODO
            // $user_id = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            return $user_id;
        } else {
            return NULL;
        }
    }

    /** Opal
     * Validating user api key
     * If the api key is there in db, it is a valid key
     * @param String $api_key user api key
     * @return boolean
     */
    public function isValidApiKey($api_key) {
        $stmt = $this->conn->prepare("SELECT emailaddress from tbl_customer WHERE api_key = ?");
        $stmt->bind_param("s", $api_key);
        $stmt->execute();
        $stmt->store_result();
        $num_rows = $stmt->num_rows;
		//echo $num_rows;
        $stmt->close();
        return $num_rows > 0;
    }

    /** Opal
     *  Generating random Unique MD5 String for user Api key 
     */
    public function generateApiKey() {
        return md5(uniqid(rand(), true));
    }
	/** Opal 
	* Update the Api Key in customer table
	**/
	public function updateApikey($emailaddress = '', $api_key){
		if($emailaddress != ""){
			$stmt = $this->conn->prepare("UPDATE tbl_customer set api_key = ? WHERE emailaddress = ?");
			$stmt->bind_param("ss", $api_key, $emailaddress);
			$result = $stmt->execute();
			$stmt->close();
		}else{
			//echo $api_key;
			$key = NULL;
			$stmt = $this->conn->prepare("UPDATE tbl_customer SET api_key = ? WHERE api_key = ?");
			$stmt->bind_param("is", $key, $api_key);			
			$result = $stmt->execute();
			$stmt->close();
			
		}
		return $result;
       
	}
    /* ------------- `tasks` table method ------------------ */

    /**
     * Creating new task
     * @param String $user_id user id to whom task belongs to
     * @param String $task task text
     */
    public function createTask($user_id, $task) {
        $stmt = $this->conn->prepare("INSERT INTO tasks(task) VALUES(?)");
        $stmt->bind_param("s", $task);
        $result = $stmt->execute();
        $stmt->close();

        if ($result) {
            // task row created
            // now assign the task to user
            $new_task_id = $this->conn->insert_id;
            $res = $this->createUserTask($user_id, $new_task_id);
            if ($res) {
                // task created successfully
                return $new_task_id;
            } else {
                // task failed to create
                return NULL;
            }
        } else {
            // task failed to create
            return NULL;
        }
    }

    /**
     * Fetching single task
     * @param String $task_id id of the task
     */
    public function getWarningLight($cms_id) {
        $stmt = $this->conn->prepare("SELECT cms_id, cms_title, cms_description, cms_icon from tbl_cms WHERE cms_id = ?");
        $stmt->bind_param("i", $cms_id);
        if ($stmt->execute()) {
            $res = array();
            $stmt->bind_result($cms_id, $cms_title, $cms_description, $cms_icon);
            // TODO
            // $task = $stmt->get_result()->fetch_assoc();
            $stmt->fetch();
            $res["cms_id"] = $cms_id;
            $res["cms_title"] = $cms_title;
            $res["cms_description"] = $cms_description;
            $res["cms_icon"] = $cms_icon;
            $stmt->close();
            return $res;
        } else {
            return NULL;
        }
    }

    /**
     * Fetching all user tasks
     * @param String $user_id id of the user
     */
    public function getWarningLights() {
        $stmt = $this->conn->prepare("SELECT * FROM tbl_cms where warning_lights = '1' AND status = '1'");
        $stmt->execute();
        $warningLights = $stmt->get_result();
        $stmt->close();
        return $warningLights;
    }

    /**
     * Updating task
     * @param String $task_id id of the task
     * @param String $task task text
     * @param String $status task status
     */
    public function updateTask($user_id, $task_id, $task, $status) {
        $stmt = $this->conn->prepare("UPDATE tasks t, user_tasks ut set t.task = ?, t.status = ? WHERE t.id = ? AND t.id = ut.task_id AND ut.user_id = ?");
        $stmt->bind_param("siii", $task, $status, $task_id, $user_id);
        $stmt->execute();
        $num_affected_rows = $stmt->affected_rows;
        $stmt->close();
        return $num_affected_rows > 0;
    }

    /**
     * Deleting a task
     * @param String $task_id id of the task to delete
     */
    public function deleteTask($user_id, $task_id) {
        $stmt = $this->conn->prepare("DELETE t FROM tasks t, user_tasks ut WHERE t.id = ? AND ut.task_id = t.id AND ut.user_id = ?");
        $stmt->bind_param("ii", $task_id, $user_id);
        $stmt->execute();
        $num_affected_rows = $stmt->affected_rows;
        $stmt->close();
        return $num_affected_rows > 0;
    }

    /* ------------- `user_tasks` table method ------------------ */

    /**
     * Function to assign a task to user
     * @param String $user_id id of the user
     * @param String $task_id id of the task
     */
    public function createUserTask($user_id, $task_id) {
        $stmt = $this->conn->prepare("INSERT INTO user_tasks(user_id, task_id) values(?, ?)");
        $stmt->bind_param("ii", $user_id, $task_id);
        $result = $stmt->execute();

        if (false === $result) {
            die('execute() failed: ' . htmlspecialchars($stmt->error));
        }
        $stmt->close();
        return $result;
    }
	
	/* ------------- `tbl_inquiry` table method ------------------ */

    /**
     * Function to assign a task to user
     * @param String $user_id id of the user
     * @param String $task_id id of the task
     */
    public function createCustomerInquiry($user_id, $description) {
		$today_date = date('Y-m-d');
        $stmt = $this->conn->prepare("INSERT INTO tbl_inquiry (sender_id, message, inquiry_date) values(?, ?, ?)");
		//print_r($stmt);
        $stmt->bind_param("iss", $user_id, $description, $today_date);
		//echo "Hello";die();
        $result = $stmt->execute();

        if (false === $result) {
            die('execute() failed: ' . htmlspecialchars($stmt->error));
        }
        $stmt->close();
        return $result;
    }
	
	/* ------------- `tbl_customer` table method ------------------ */

    /**
     * Function get All data of Customer
     * @param String $user_id id of the user
     */
    public function getCustomer($user_id) {
        $stmt = $this->conn->prepare("SELECT * FROM `tbl_customer` WHERE user_id = ?");
        $stmt->bind_param("i", $user_id);
        $result = $stmt->execute();

        if (false === $result) {
            die('execute() failed: ' . htmlspecialchars($stmt->error));
        }
		$response = $stmt->get_result();
        $stmt->close();
        return $response;
    }
	public function getCustomerVehicle($vehicle_id) {
        $stmt = $this->conn->prepare("SELECT * FROM `tbl_vehicle` WHERE vehicle_id = ?");
        $stmt->bind_param("i", $vehicle_id);
        $result = $stmt->execute();

        if (false === $result) {
            die('execute() failed: ' . htmlspecialchars($stmt->error));
        }
		$response = $stmt->get_result();
        $stmt->close();
        return $response;
    }
	/**
     * Fetching all Customer Cars
     * @param String $user_id id of the user
     */
    public function getCustomerCar($user_id) {
		$today_date = date('Y-m-d');
        $stmt = $this->conn->prepare("SELECT * FROM `tbl_vehicle` as v, tbl_vehicleplandetail as vp WHERE v.vehicle_id = vp.vehicle_id AND v.user_id = ? AND vp.end_date >= ?");
        $stmt->bind_param("is", $user_id, $today_date );
        $result = $stmt->execute();
        $cars_data = $stmt->get_result();
		$final_cars = array();
		while ($cars =  $cars_data->fetch_assoc()) {
                $tmp = array();
				$bank_detail = $this->getBankAccount($cars['vehicle_id']);
				//print_r($bank_detail);
				//die();
				$tmp["payment_frequency"] = $cars['paymentfrequency'];
				$tmp["payment_day"] = $cars['paymentday'];
				$tmp["payment_amount"] = $bank_detail['ddamount'];
				$plan_detail = $this->getPlan($cars['plan_id']);
                $tmp["plateno"] = strtoupper($cars["plateno"]);
                $tmp["make"] = $cars["make"];
                $tmp["model"] = $cars["model"];
				$tmp["year"] = $cars["year"];
				$tmp["color"] = $cars["color"];
				$tmp["vehicle_id"] = $cars['vehicle_id'];
				//$date_diff = date_diff($next_serrvice, date_create($today_date));
				$nodays=(strtotime($cars["next_service_date"]) - strtotime($today_date))/ (60 * 60 * 24); 
				$tmp["nextservice"] = (int)$nodays;
				if($nodays >= 0){
					$tmp["nextservicetext"] = "Before Service";
				}else{
					$tmp["nextservicetext"] = "Overdue";
				}
				$tmp["plan_id"] = $cars['plan_id'];
				$tmp["plan_type"] = $plan_detail['planname'];
				$tmp["term"] = $plan_detail['duration'];
				
				$tmp["renewal_date"] = date("d-m-Y", strtotime($cars["end_date"]));
				
                array_push($final_cars, $tmp);
            }
        $stmt->close();
        return $final_cars;
    }
	
	/**
     * Get Last Booking Details 
     * @param String $car_id of the user
     */
    public function getLastBooking($car_id) {
		$stmt = $this->conn->prepare("SELECT * FROM `tbl_booking` WHERE vehicle_id = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $stmt->store_result();
        $num_rows = $stmt->num_rows;
		if($num_rows > 0){
			
		}else{
			
		}

    }

	public function getTotalService($vehicle_id){
		$stmt = $this->conn->prepare("SELECT count(*) as b FROM `tbl_booking` WHERE vehicle_id = ?");
        $stmt->bind_param("i", $vehicle_id);
		$result = $stmt->execute();
        $cars_data = $stmt->get_result();
		while ($cars =  $cars_data->fetch_assoc()) {
			$total_number = $cars['b'];
		}

		return $total_number;
	}
	public function GetServiceDuration($service_id) {
		$stmt = $this->conn->prepare("SELECT duration FROM tbl_servicetype WHERE servicetype_id = ?");
        $stmt->bind_param("i", $service_id);
        if ($stmt->execute()) {
            $stmt->bind_result($duration);
            $stmt->fetch();
            // TODO
            // $user_id = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            return $duration;
        } else {
            return NULL;
        }

    }
	public function getServiceId($plan_id, $getTotalService) {
		$stmt = $this->conn->prepare("SELECT servicetype_id FROM tbl_plandetails WHERE plan_id = ? ORDER BY plandetail_id ASC LIMIT ?,1");
        $stmt->bind_param("ii", $plan_id, $getTotalService);
        if ($stmt->execute()) {
			
            $stmt->bind_result($servicetype_id);
            $stmt->fetch();

			//ECHO $service_id;DIE();
            // TODO
            // $user_id = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            return $servicetype_id;
        } else {
            return NULL;
        }

    }
	public function insertBooking($service_id, $user_id, $vehicle_id, $booking_time, $location_address, $pickup_time, $dropoff_time, $duration, $booking_end_time){
		$stmt = $this->conn->prepare("INSERT INTO tbl_booking (vehicle_id, user_id, servicetype_id, location, carpickuptime, cardropofftime, bookingdatetime, bookingendtime, duration) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
		//echo "".$user_id."=".$lat."=".$lan."=".$datetime."=".$api_key;
		$stmt->bind_param("iiissssss", $vehicle_id, $user_id, $service_id, $location_address, $pickup_time, $dropoff_time, $booking_time, $booking_end_time, $duration );
        $result = $stmt->execute();
		$booking_id = $stmt->insert_id;
        $stmt->close();



		//Insert in to jobcard
		$stmt = $this->conn->prepare("INSERT INTO tbl_jobcard (booking_id) VALUES (?)");
		//echo "".$user_id."=".$lat."=".$lan."=".$datetime."=".$api_key;
		$stmt->bind_param("i", $booking_id);
        $result = $stmt->execute();
		$job_card_id = $stmt->insert_id;
        $stmt->close();

		$allservices_query = $this->conn->prepare("SELECT * FROM tbl_servicechecklist WHERE servicetype_id = ? AND parent_id = 0 "); 
        $allservices_query->bind_param("i",$service_id);
        $allservices_query->execute();
        $allservices_query_result = $allservices_query->get_result();
        while($allservices_query_data = $allservices_query_result->fetch_assoc()){
            $allservices[]  = $allservices_query_data;
        }
        if(! empty($allservices)){
            foreach($allservices as $servicetype){
                $parent_id = $servicetype['parent_id'];
                if($parent_id == 0){
						
					//Insert in to jobcard
					$stmt1 = $this->conn->prepare("INSERT INTO tbl_jobcard_detail (jobcard_id, service_type_name, parent_id) VALUES (?,?,?)");
					$stmt1->bind_param("isi", $job_card_id, $servicetype['name'], $parent_id);
					$result = $stmt1->execute();
					$job_card_detail_id = $stmt1->insert_id;
					$stmt1->close();
						
                    $servicechecklist_id = $servicetype['servicechecklist_id'];
                    // here we fetch all sub category
                    $allsubservice_query = $this->conn->prepare("SELECT *, GROUP_CONCAT(checklist_value) as options FROM tbl_servicechecklist as s, tbl_servicechecklistdetail as sd WHERE s.servicechecklist_id=sd.servicechecklist_id AND parent_id = ? GROUP BY sd.servicechecklist_id "); 
                    $allsubservice_query->bind_param("i",$servicechecklist_id);
                    $allsubservice_query->execute();
                    $allsubservice_result = $allsubservice_query->get_result();
					$l = 0;
                    while($allsubservice_data  = $allsubservice_result->fetch_assoc()){
                        //Insert in to jobcard_Detail
						$stmt2 = $this->conn->prepare("INSERT INTO tbl_jobcard_detail (jobcard_id, service_type_name, parent_id, service_type_options) VALUES (?,?,?,?)");
						$stmt2->bind_param("isis", $job_card_id, $allsubservice_data['name'], $job_card_detail_id, $allsubservice_data['options']);
						$result = $stmt2->execute();
						$stmt2->close();
                            
                    }   
                    // End 
                }
            }
        }   

        if ($result) {
            return true;
        } else {
            return NULL;
        }
	}
	public function getAvailableDateTime($date, $current_user){
		$stmt = $this->conn->prepare("SELECT bookingdatetime, bookingendtime, vehicle_id, user_id FROM `tbl_booking` WHERE bookingdatetime LIKE '".$date."%'");
        //$stmt->bind_param("s", $date);
        $stmt->execute();
        $bank_data = $stmt->get_result();
		$return_arrray=array();
        while ($bank = $bank_data->fetch_assoc()) {
			$tmp = array();
			
			if($bank['user_id'] == $current_user){
				$customerData = $this->getCustomer($bank['user_id'] );
				$vehicleData = $this->getCustomerVehicle($bank['vehicle_id'] );
				while ($customer = $customerData->fetch_assoc()) {
					$customer_name =  $customer['firstname']." ".$customer['lastname'];
				}
				while ($vehicle = $vehicleData->fetch_assoc()) {
					$vehicle_plate =  $vehicle['plateno'];
				}
				
				$color = 'yellow';
				$title = $customer_name. ' - '.$vehicle_plate;
			}else{
				$color = 'grey';
				$title = 'Taken';
			}
			$tmp['title'] = $title;
			$tmp['color'] = $color;
			$tmp['starttime'] = $bank['bookingdatetime'];
			$tmp['endtime'] = $bank['bookingendtime'];
			array_push($return_arrray, $tmp);
		}
		//print_r($return_arrray);die();
		return $return_arrray;
	}

	public function getMyService($user_id){
	//echo "SELECT j.scheduledatetime, v.plateno, v.model, v.make, jc.jobcard_pdf  FROM `tbl_jobs` as j, tbl_jobcard as jc, tbl_booking as b, tbl_vehicle as v WHERE b.vehicle_id = v.vehicle_id AND j.booking_id = b.booking_id AND jc.booking_id = b.booking_id  AND j.status = 4 AND  v.user_id = ".$user_id;
		$stmt = $this->conn->prepare("SELECT b.scheduledatetime, v.plateno, v.model, v.make, jc.jobcard_pdf  FROM tbl_jobcard as jc, tbl_booking as b, tbl_vehicle as v WHERE b.vehicle_id = v.vehicle_id  AND jc.booking_id = b.booking_id  AND b.job_status = 4 AND  v.user_id = ".$user_id);
        //$stmt->bind_param("s", $date);
        $stmt->execute();
        $bank_data = $stmt->get_result();
		$return_arrray=array();
        while ($bank = $bank_data->fetch_assoc()) {
			$tmp = array();
			$date = new DateTime($bank['scheduledatetime']);
			
			$tmp['model'] = $bank['plateno']." / ".$bank['make']." / ".$bank['model'];
			$tmp['date'] = $date->format('d/m/Y');
			$tmp['url'] = $bank['jobcard_pdf'];
			array_push($return_arrray, $tmp);
		}
		//print_r($return_arrray);die();
		return $return_arrray;
	}

	public function getMyBooking($user_id){
	//echo "SELECT b.bookingdatetime, v.plateno, v.model, v.make  FROM `tbl_booking` as b, tbl_vehicle as v WHERE b.vehicle_id = v.vehicle_id  AND v.user_id = ".$user_id;
		$stmt = $this->conn->prepare("SELECT b.bookingdatetime, v.plateno, v.model, v.make  FROM `tbl_booking` as b, tbl_vehicle as v WHERE b.vehicle_id = v.vehicle_id  AND v.user_id = ".$user_id);
        //$stmt->bind_param("s", $date);
        $stmt->execute();
        $bank_data = $stmt->get_result();
		$return_arrray=array();
        while ($bank = $bank_data->fetch_assoc()) {
			$tmp = array();
			$date = new DateTime($bank['bookingdatetime']);
			
			$tmp['model'] = $bank['plateno']." / ".$bank['make']." / ".$bank['model'];
			$tmp['date'] = $date->format('d/m/Y');
			$tmp['time'] = $date->format('H:i');
			array_push($return_arrray, $tmp);
		}
		//print_r($return_arrray);die();
		return $return_arrray;
	}
	/**
     * Get Bank Account Details 
     * @param String $car_id of the user
     */
    public function getBankAccount($car_id) {
		$stmt = $this->conn->prepare("SELECT * FROM `tbl_bankaccountdetail` WHERE vehicle_id = ?");
        $stmt->bind_param("i", $car_id);
        $stmt->execute();
        $bank_data = $stmt->get_result();
		$return_arrray=array();
        while ($bank = $bank_data->fetch_assoc()) {
			$return_arrray['ddamount'] = $bank['ddamount'];
			$return_arrray['paymentfrequency'] = $bank['paymentfrequency'];
			$return_arrray['paymentday'] = $bank['paymentday'];
		}
		//print_r($return_arrray);die();
		return $return_arrray;

    }
	public function getPlan($plan_id){
		$stmt = $this->conn->prepare("SELECT * FROM `tbl_plan` WHERE plan_id = ?");
        $stmt->bind_param("i", $plan_id);
        $stmt->execute();
		$plan_data = $stmt->get_result();
		$return_arrray=array();
        while($plan = $plan_data->fetch_assoc()) {
			$return_arrray['planname'] = $plan['planname'];
			$return_arrray['duration'] = $plan['duration'];
		}
		//print_r($return_arrray);
		return $return_arrray;
	}
	
	  /* ------------- `tbl_user_car_location` table method ------------------ */

    /**
     * Creating New Location
     * @param String $user_id user id to whom task belongs to
     * @param String $task task text
     */
    public function createLockLocation($user_id, $lat, $lan, $datetime, $api_key) {
		
        $stmt = $this->conn->prepare("INSERT INTO tbl_user_car_location (user_id, latitude, longitude, end_datetime, authorization_key) VALUES (?, ?, ?, ?, ?)");
		//echo "".$user_id."=".$lat."=".$lan."=".$datetime."=".$api_key;
		$stmt->bind_param("issss", $user_id, $lat, $lan, $datetime, $api_key);
        $result = $stmt->execute();
		

        $stmt->close();

        if ($result) {
            return true;
        } else {
            return NULL;
        }
    }
	/** Opal
     * Check if there is car already registred
     * If the api key is there in db
     * @param String $api_key user api key
     * @return boolean
     */
    public function getCarLocation($user_id, $api_key) {
        $stmt = $this->conn->prepare("SELECT * from tbl_user_car_location WHERE authorization_key = ? AND user_id = ? ORDER BY user_car_location_id DESC LIMIT 0,1");
        $stmt->bind_param("si", $api_key, $user_id);
        $stmt->execute();
        $result = $stmt->get_result();
        
		//echo $num_rows;
        $stmt->close();
        return $result;
    }
	function updateCustomerPhoto($user_id, $name)
	{
		$stmt = $this->conn->prepare("UPDATE  tbl_customer set profilepic = ? WHERE user_id = ?");
        $stmt->bind_param("si", $name, $user_id);
        $stmt->execute();
        $num_affected_rows = $stmt->affected_rows;
        $stmt->close();
        return $num_affected_rows > 0;
	}
	function updateCustomer($user_id, $addressLine1, $addressLine2, $mobile, $email){
		$result = $this->getCustomer($user_id);
		$data = $result->fetch_assoc();
		$user_type = 0;
		//print_r($data);
		//die();
		$stmt = $this->conn->prepare("INSERT INTO tbl_contactdetails (user_id, address1, address2, mobile, email, user_type) VALUES (?, ?, ?, ?, ?, ?) ");
		$stmt->bind_param("issssi", $user_id, $addressLine1, $addressLine2, $mobile, $email, $user_type);
        $stmt->execute();
		$stmt->close();
		
		
		$stmt = $this->conn->prepare("UPDATE  tbl_customer set address = ?, address2 = ?, emailaddress = ?, mobilenumber= ? WHERE user_id = ?");
        $stmt->bind_param("ssssi", $addressLine1, $addressLine2, $email, $mobile, $user_id);
        $stmt->execute();
        $num_affected_rows = $stmt->affected_rows;
        $stmt->close();
        return $num_affected_rows > 0;
	}
	function getSitemanagerByCustomer($user_id){
		
		$stmt = $this->conn->prepare("SELECT addedbyuser_id FROM `tbl_customer` WHERE user_id = ?");
        $stmt->bind_param("i", $user_id);
        $result = $stmt->execute();

        if (false === $result) {
            die('execute() failed: ' . htmlspecialchars($stmt->error));
        }
		$customer_data_res = $stmt->get_result();
		$customer_data = $customer_data_res->fetch_assoc();
        $stmt->close();
		//print_r($customer_data);die();
		$stmt = $this->conn->prepare("SELECT addedbyuser_id,usertype_id FROM `tbl_adminuser` WHERE user_id = ?");
        $stmt->bind_param("i", $customer_data['addedbyuser_id']);
        $result = $stmt->execute();

        if (false === $result) {
            die('execute() failed: ' . htmlspecialchars($stmt->error));
        }
		$customer_sales_man_res = $stmt->get_result();
        $stmt->close();
		$customer_sales_man = $customer_sales_man_res->fetch_assoc();
		if($customer_sales_man['usertype_id'] == "1"){

			$stmt = $this->conn->prepare("SELECT addedbyuser_id,usertype_id FROM `tbl_adminuser` WHERE user_id = ?");
			$stmt->bind_param("i", $customer_sales_man['addedbyuser_id']);
			$result = $stmt->execute();
			if (false === $result) {
				die('execute() failed: ' . htmlspecialchars($stmt->error));
			}
			$customer_sales_manager_res = $stmt->get_result();
			$customer_sales_manager = $customer_sales_manager_res->fetch_assoc();
			$stmt->close();
			$sitemanager_id = $customer_sales_manager['addedbyuser_id'];
		}else{
			$sitemanager_id = $customer_sales_man['addedbyuser_id'];
		}
		return $sitemanager_id;
	}
	
	public function getUserFeedback($user_id){
		$sender_id = $user_id;
		$receiver_id = $this->getSitemanagerByCustomer($user_id);
		
		$stmt = $this->conn->prepare("SELECT conversation_id FROM `tbl_conversation` WHERE ((sender_id = ? AND receiver_id = ?) OR (sender_id = ? AND receiver_id = ?)) AND status = 0");
        $stmt->bind_param("iiii", $sender_id, $receiver_id, $receiver_id, $sender_id);
        $result = $stmt->execute();

        if (false === $result) {
            die('execute() failed: ' . htmlspecialchars($stmt->error));
        }
		$check_conversation_res = $stmt->get_result();
		$check_conversation = $check_conversation_res->fetch_assoc();

		//print_r($check_conversation);
        $stmt->close();
		if(!empty($check_conversation)){

			$conversation_id = $check_conversation['conversation_id'];
			$stmt = $this->conn->prepare("SELECT * FROM `tbl_message` WHERE conversation_id = ? ORDER BY conversation_datetime ASC");
			$stmt->bind_param("i", $conversation_id);
			$result = $stmt->execute();

			if (false === $result) {
				die('execute() failed: ' . htmlspecialchars($stmt->error));
			}
			$conversation = $stmt->get_result();
			$stmt->close();
			return $conversation;
		}		
	}
	public function getAdminUser($admin_id){
		$stmt = $this->conn->prepare("SELECT * FROM `tbl_adminuser` WHERE user_id = ?");
		$stmt->bind_param("i", $admin_id);
		$result = $stmt->execute();
		if (false === $result) {
			die('execute() failed: ' . htmlspecialchars($stmt->error));
		}
		$customer_sales_manager_res = $stmt->get_result();
		$customer_sales_manager = $customer_sales_manager_res->fetch_assoc();
		return $customer_sales_manager;
	}
	function insertMessage($sender_id, $receiver_id, $message, $conversation_id, $conversation_datetime, $send_by){
		$stmt = $this->conn->prepare("INSERT INTO `tbl_message`(`conversation_id`, `sender_id`, `receiver_id`, `message`, `send_by`, `conversation_datetime`) VALUES (?, ?, ?, ?, ?, ?)");
		$stmt->bind_param("iiisss", $conversation_id, $sender_id, $receiver_id, $message, $send_by, $conversation_datetime);
        $result = $stmt->execute();
        $stmt->close();

        if ($result) {
            return true;
        } else {
            return NULL;
        }
	}
	function insertconversion($sender_id, $receiver_id, $conversation_datetime, $status){
		$stmt = $this->conn->prepare("INSERT INTO `tbl_conversation`(`sender_id`, `receiver_id`, `conversation_datetime`, `status`) VALUES (?, ?, ?, ?)");
		$stmt->bind_param("iiss", $sender_id, $receiver_id, $conversation_datetime, $status);
        $result = $stmt->execute();
		$last_id = $stmt->insert_id;
        $stmt->close();

        if ($result) {
			//$id = $this->conn->lastInsertId();
            return $last_id;
        } else {
            return NULL;
        }
	}
}
?>
