<?php

/**
 * Class to handle all db operations
 * This class will have CRUD methods for database tables
 *
 * @author Ravi Tamada
 * @link URL Tutorial link
 */
class DbHandler {
    private $conn;
    function __construct() {
        require_once dirname(__FILE__) . '/DbConnect.php';
        // opening db connection
        $db = new DbConnect();
        $this->conn = $db->connect();
    }
    /** Opal
     * Checking user login
     * @param String $email User login email id
     * @param String $password User login password
     * @return boolean User login status success/fail
     */
    public function checkLogin($email, $password) {
        // fetching user by email
        $stmt = $this->conn->prepare("SELECT password FROM tbl_adminuser WHERE emailaddress = ? AND status = 1");

        $stmt->bind_param("s", $email);

        $stmt->execute();

        $stmt->bind_result($password_hash);

        $stmt->store_result();

        if ($stmt->num_rows > 0) {
            // Found user with the email
            // Now verify the password

            $stmt->fetch();

            $stmt->close();

            if (PassHash::check_password($password_hash, $password)) {
                // User password is correct
                return TRUE;
            } else {
                // user password is incorrect
                return FALSE;
            }
        } else {
            $stmt->close();

            // user not existed with the email
            return FALSE;
        }
    }

    /** Opal
     * Checking for duplicate user by email address
     * @param String $email email to check in db
     * @return boolean
     */
    private function isUserExists($email) {
        $stmt = $this->conn->prepare("SELECT id from tbl_customer WHERE emailaddress = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $stmt->store_result();
        $num_rows = $stmt->num_rows;
        $stmt->close();
        return $num_rows > 0;
    }



    /** Opal
     * Fetching user by email
     * @param String $email User email id
     */
    public function getUserByEmail($email) {
        $stmt = $this->conn->prepare("SELECT firstname, lastname, emailaddress, api_key, profilepic FROM tbl_adminuser WHERE emailaddress = ?");
        $stmt->bind_param("s", $email);
		//print_r($stmt);
        if ($stmt->execute()) {
            // $user = $stmt->get_result()->fetch_assoc();
            $stmt->bind_result($firstname, $lastname, $emailaddress, $api_key, $profilepic);
            $stmt->fetch();																									
            $user = array();
            $user["firstname"] = $firstname;
            $user["lastname"] = $lastname;
            $user["emailaddress"] = $emailaddress;
            $user["api_key"] = $api_key;
			if($profilepic == NULL || $profilepic == ""){
				$user["userImage"] = 'no-profile.jpg';
			}else{
				$user["userImage"] = $profilepic;
			}
			//print_r($user);
            $stmt->close();
            return $user;
        } else {
            return NULL;
        }
    }

    /** Opal
     * Fetching user api key
     * @param String $user_id user id primary key in user table
     */
    public function getApiKeyById($user_id) {
        $stmt = $this->conn->prepare("SELECT api_key FROM tbl_customer WHERE id = ?");
        $stmt->bind_param("i", $user_id);
        if ($stmt->execute()) {
            // $api_key = $stmt->get_result()->fetch_assoc();
            // TODO
            
            $stmt->bind_result($api_key);
            
            $stmt->close();
            return $api_key;
        } else {
            return NULL;
        }
    }
	  /** Opal
     * Fetching user api key
     * @param String $user_id user id primary key in user table
     */
    public function getApiKeyByEmail($emailaddress) {
        $stmt = $this->conn->prepare("SELECT api_key FROM tbl_adminuser WHERE emailaddress = ?");
        $stmt->bind_param("s", $emailaddress);
        if ($stmt->execute()) {
            // $api_key = $stmt->get_result()->fetch_assoc();
            // TODO
            $stmt->bind_result($api_key);
            $stmt->fetch();	
            $stmt->close();
            return $api_key;
        } else {
            return NULL;
        }
    }
    public function search_data($searchString){
        $param = "%{$searchString}%";
        $stmt = $this->conn->prepare("SELECT c.user_id,c.firstname,c.lastname,c.mobilenumber,v.plateno from tbl_customer as c left join tbl_vehicle as v ON c.user_id = v.user_id where c.firstname like ? OR c.lastname like ? OR c.mobilenumber LIKE ? OR v.plateno LIKE ?  OR v.make LIKE ?  OR v.model LIKE ? ");
        $stmt->bind_param("ssssss", $param,$param,$param,$param,$param,$param);
        $result = $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }

    public function store_incident($user_id,$incidentdate,$incidenttime,$incidentlocation,$incidentperson,$incidentdetails){
        $current_date_time  = date("Y-m-d h:i:s");
        $incident_date_time = $incidentdate.$incidenttime;
        $store_incident = date("Y-m-d h:i:s",strtotime($incident_date_time));

        $stmt = $this->conn->prepare("INSERT INTO tbl_incidentreport (admin_id,incident_datetime,incident_location,incident_people,incident_details,report_datetime) VALUES (?,?,?,?,?,?)");
        $stmt->bind_param("isssss",$user_id,$store_incident,$incidentlocation,$incidentperson,$incidentdetails,$current_date_time);
        if($stmt->execute()){
            $stmt->close();
            return 0;
        }else{
            $stmt->close();
            return 1;
        }
    }

    public function store_booking_info($booking_id,$technician_id){
        $job_status = 0;
        $stmt = $this->conn->prepare("UPDATE tbl_booking SET job_status = ? , technician_id = ? WHERE booking_id = ?");
        $stmt->bind_param("iii",$job_status,$technician_id,$booking_id);
        if($stmt->execute()){
            $stmt->close();
            return 1;
        }else{
            $stmt->close();
            return 0;
        }


    }


    public function getBookCalendar($date){
            //echo "SELECT b.bookingdatetime, b.vehicle_id, b.bookingendtime, v.plateno, v.model, v.make, b.booking_id, b.servicetype_id  FROM tbl_jobcard as jc, tbl_booking as b, tbl_vehicle as v WHERE b.vehicle_id = v.vehicle_id AND j.booking_id = b.booking_id  AND  b.bookingdatetime LIKE '".$date."%' and b.technician_id =".$user_id." GROUP BY b.booking_id";
            $stmt = $this->conn->prepare("SELECT b.bookingdatetime, b.vehicle_id, b.bookingendtime, b.technician_id ,v.plateno, v.model, v.make, b.booking_id, b.servicetype_id , b.user_id as customerid FROM tbl_jobcard as jc, tbl_booking as b, tbl_vehicle as v WHERE b.vehicle_id = v.vehicle_id AND  b.bookingdatetime LIKE '".$date."%'  GROUP BY b.booking_id"); //and b.technician_id =".$user_id."
            //$stmt->bind_param("s", $date);
            $stmt->execute();
            $bank_data = $stmt->get_result();
            $return_arrray=array();
            while ($bank = $bank_data->fetch_assoc()) {
                $tmp = array();
                $tmp['starttime'] = $bank['bookingdatetime'];
                $tmp['endtime'] = $bank['bookingendtime'];
                $tmp['plateno'] = $bank['plateno'];
                $tmp['model'] = $bank['model'];
                $tmp['make'] = $bank['make'];
                $tmp['booking_id'] = $bank['booking_id'];
                $tmp['customerid'] = $bank['customerid'];
                $tmp['technician_id'] = $bank['technician_id'];
                


                
                $service_type_query =  $this->conn->prepare("SELECT name FROM tbl_servicetype WHERE servicetype_id = ? "); 
                $service_type_query->bind_param("i",$bank['servicetype_id']);
                $service_type_query->execute();
                $service_type_query_result = $service_type_query->get_result();
                $service_type_query_data  = $service_type_query_result->fetch_assoc();
                $service_name = $service_type_query_data['name'];
                $tmp['service_type'] = $service_name;
                $tmp['booking_id'] = $bank['booking_id'];
                $tmp['vehicle_id'] = $bank['vehicle_id'];
                array_push($return_arrray, $tmp);
            }
            //print_r($return_arrray);die();
            return $return_arrray;
        }


    public function updateStaffStatusSign($sign, $user_id){
		$stmt = $this->conn->prepare("UPDATE tbl_bankaccountdetail SET status= 1 WHERE user_id = ? AND vehicle_id = 0");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
		$registration_date = date('Y-m-d');
		$registration_end_date = date('Y-m-d', strtotime('+1 years'));
		$stmt = $this->conn->prepare("UPDATE tbl_adminuser SET signature_proof= ?, registrationend_date = ?, registration_date = ?, status= 1  WHERE user_id = ?");
        $stmt->bind_param("sssi", $sign, $registration_date, $registration_end_date, $user_id);
        $stmt->execute();
        $num_affected_rows = $stmt->affected_rows;
        $stmt->close();
        return $num_affected_rows > 0;
	}

    public function insertTechnician($first_name, $last_name, $date_of_birth, $gender, $mobile_no, $adddress, $address2, $email_address, $driver_licence_no, $driver_licence_expiry_date, $bank_account_name, $bank_name, $payment_day, $payment_frequency, $bank_account_no, $user_id, $drive_proof1, $drive_proof2, $profilepic,$hourly_rate){
		$date = date('Y-m-d H:i:s');


        // insert query
		$user_type_id = 4; // for technician
		$status = 2;
        $stmt = $this->conn->prepare("INSERT INTO tbl_adminuser(`addedbyuser_id`, `usertype_id`, `firstname`, `lastname`, `emailaddress`, `dateofborth`, `mobilenumber`, `address`, `address2`, `drivinglicencenumber`, `drivinglicenceexpiredate`, `drivinglicencefile1`, `drivinglicencefile2`, `profilepic`, `gender`, `status`, registration_date) values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("iisssssssssssssis", $user_id, $user_type_id, $first_name, $last_name, $email_address, $date_of_birth, $mobile_no, $adddress, $address2, $driver_licence_no, $driver_licence_expiry_date, $drive_proof1, $drive_proof2, $profilepic, $gender, $status, $date);

        $result = $stmt->execute();
		//print_r($result);
		$last_id = $stmt->insert_id;
        $stmt->close();

		$user_type = 1;
		$vehicle_id = 0;

		$stmt = $this->conn->prepare("INSERT INTO tbl_bankaccountdetail(`user_id`, `vehicle_id`, `accountname`, `accountnumber`, `bankname`, `paymentfrequency`, `paymentday`, `status`, `user_type`,`hourly_rate`) values (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("iissssssss", $last_id, $vehicle_id, $bank_account_name, $bank_account_no, $bank_name, $payment_frequency, $payment_day, $status, $user_type,$hourly_rate);

        $result = $stmt->execute();
        $stmt->close();

        // Check for successful insertion
        if ($result) {
            // User successfully inserted
            return $last_id;
        } else {
            // Failed to create user
            return USER_CREATE_FAILED;
        }
    }
    
    public function updateTechnician($first_name, $last_name, $date_of_birth, $gender, $mobile_no, $adddress, $address2, $email_address, $driver_licence_no, $driver_licence_expiry_date, $bank_account_name, $bank_name, $payment_day, $payment_frequency, $bank_account_no, $user_id, $drive_proof1, $drive_proof2, $profilepic, $update_user_id,$hourly_rate){
		$date = date('Y-m-d H:i:s');
            // insert query
			$user_type_id = 4; // for technician
			$status = 2;
            $stmt = $this->conn->prepare("UPDATE tbl_adminuser SET addedbyuser_id = ?, usertype_id= ?, firstname= ?, lastname=?, emailaddress=?, dateofborth=?, mobilenumber=?, address=?, address2=?, drivinglicencenumber=?, drivinglicenceexpiredate=?, drivinglicencefile1=?, drivinglicencefile2=?, profilepic=?, gender=?, status=?, registration_date=? WHERE user_id = ?");
            $stmt->bind_param("iisssssssssssssssi", $user_id, $user_type_id, $first_name, $last_name, $email_address, $date_of_birth, $mobile_no, $adddress, $address2, $driver_licence_no, $driver_licence_expiry_date, $drive_proof1, $drive_proof2, $profilepic, $gender, $status, $date, $update_user_id);
            $result = $stmt->execute();
            $stmt->close();

			$user_type = 1;
			$vehicle_id = 0;

			$stmt = $this->conn->prepare("UPDATE tbl_bankaccountdetail SET accountname=?, accountnumber=?, bankname=?, paymentfrequency=?, paymentday=?, status=?, user_type=? , hourly_rate = ? WHERE user_id = ? AND vehicle_id = 0");
            $stmt->bind_param("sssssssis",  $bank_account_name, $bank_account_no, $bank_name, $payment_frequency, $payment_day, $status, $user_type,$hourly_rate,$update_user_id);

            $result = $stmt->execute();
            $stmt->close();

            // Check for successful insertion
            if ($result) {
                // User successfully inserted
                return $update_user_id;
            } else {
                // Failed to create user
                return USER_CREATE_FAILED;
            }
	}

    public function store_improvement($user_id,$improvementdate,$shortcomingdetail,$proposedplan,$recorddetail){
        $stmt = $this->conn->prepare("INSERT INTO tbl_recordimprovement (sitemanager_id,shortcomingdetail,proposedplan,recorddetail,recorddate) VALUES (?,?,?,?,?)");    
        $stmt->bind_param("issss",$user_id,$shortcomingdetail,$proposedplan,$recorddetail,$improvementdate);
        if($stmt->execute()){
            $stmt->close();
            return 1;
        }else{
            $stmt->close();
            return 0;
        }
    }

    public function get_pending_improvements($userid){
        $stmt = $this->conn->prepare("SELECT * FROM tbl_recordimprovement  WHERE sitemanager_id = ? AND status = 0 ");
        $stmt->bind_param("i",$userid);
        $data = $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }
    public function update_opening_hours($user_id,$mon_to_fri_time,$saturday_time,$holidaysdata){
        // remove all holidays 
        $delete_exe = $this->conn->prepare("DELETE FROM tbl_publicholidays WHERE site_manager_id = ?");
        $delete_exe->bind_param("i",$user_id);
        $delete_exe->execute();
        $delete_exe->close();
        // End 

        // Here we check info exist or not
        $chk_data = $this->conn->prepare("SELECT id FROM tbl_openinghours WHERE site_manager_id = ?");
        $chk_data->bind_param("i",$user_id);
        $data = $chk_data->execute();
        $result = $chk_data->get_result();
        $chk_data->close();
        if($result->num_rows > 0){
           // here we update the information 
            $update_stat = $this->conn->prepare("UPDATE tbl_openinghours SET mon_to_fri_time = ? ,saturday_time = ? WHERE site_manager_id = ?");
            $update_stat->bind_param('ssi',$mon_to_fri_time,$saturday_time,$user_id);
            $update_stat->execute();
            $update_stat->close();
        }else{
            // insert data
            $insert_info = $this->conn->prepare("INSERT INTO tbl_openinghours (site_manager_id,mon_to_fri_time,saturday_time) VALUES(?,?,?)");
            $insert_info->bind_param('iss',$user_id,$mon_to_fri_time,$saturday_time);
            $insert_info->execute();
            $insert_info->close();
        }
        // here we insert holidays data is entered
        if(count($holidaysdata) > 0){
            foreach($holidaysdata as $holidayinfo){
                $date = $holidayinfo['holidayFromdate'].":::".$holidayinfo['holidayTodate'];
                $holiday_statm = $this->conn->prepare("INSERT INTO tbl_publicholidays (site_manager_id,publicholidays_name,publicholidays_fromdate_todate) VALUES (?,?,?)");
                $holiday_statm->bind_param("iss",$user_id,$holidayinfo['holidayName'],$date);
                $holiday_statm->execute();
                $holiday_statm->close();
            }
        }
        // end 
        return 1;
    }

    public function get_opening_hours_info($userid){
        $stmt = $this->conn->prepare("select * from tbl_openinghours where site_manager_id = ?");
        $stmt->bind_param("i",$userid);
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }

    public function get_opening_hours_holidays_info($userid){
        $stmt = $this->conn->prepare("select * from tbl_publicholidays where site_manager_id = ?");
        $stmt->bind_param("i",$userid);
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }

    public function store_request_purchase($user_id,$improvementdate,$proposedpurchase,$proposedpurchasereason,$requestpurdetail){
        $stmt = $this->conn->prepare("INSERT INTO tbl_requestpurchase (user_id,proposedpurchase,proposedpurchasereason,requestpurdetail,recorddate) VALUES (?,?,?,?,?)");
        $stmt->bind_param("issss",$user_id,$proposedpurchase,$proposedpurchasereason,$requestpurdetail,$improvementdate);
        if($stmt->execute()){
            $stmt->close();
            return 1;
        }else{
            $stmt->close();
            return 0;
        }
    }

    public function store_notes_default($user_id,$notes,$paymentid){
        $stmt = $this->conn->prepare("INSERT INTO tbl_default_report_details (user_id,bank_payment_id,notes) VALUES (?,?,?)");
        $stmt->bind_param("iis",$user_id,$paymentid,$notes);
        if($stmt->execute()){
            $stmt->close();
            return 1;
        }else{
            $stmt->close();
            return 0;
        }
    }

    public function update_user_status($userid){
        $stmt = $this->conn->prepare("UPDATE tbl_customer SET status = 4 WHERE user_id = ?");
        $stmt->bind_param("i",$userid);
        if($stmt->execute()){
            $stmt->close();
            return 1;
        }else{
            $stmt->close();
            return 0;
        }
    }

    public function get_purchase_information($user_id){
        $stmt = $this->conn->prepare("SELECT requestpurchase_id,proposedpurchase,proposedpurchasereason,requestpurdetail,recorddate,(CASE WHEN status = 1 THEN 'Pending' WHEN status = 0 THEN 'Approved' END) as mystatus  FROM tbl_requestpurchase WHERE user_id = ?");
        $stmt->bind_param("i",$user_id);
        $result = $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }

    public function getall_incident_logs($user_id){
        $stmt = $this->conn->prepare("SELECT incidentreport_id,incident_datetime,incident_location,incident_people FROM tbl_incidentreport WHERE admin_id = ?");
        $stmt->bind_param("i",$user_id);
        $result = $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }

    public function get_new_customer(){
        $stmt = $this->conn->prepare("SELECT * FROM tbl_customer WHERE MONTH(registration_date) = MONTH(CURRENT_DATE - INTERVAL 1 MONTH) AND status = 1");
        $result = $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }

    public function incident_detail_data($incidentid){
        $all_conversations = $this->conn->prepare("select d.*,u.firstname as sendername,a.firstname as receivername,u.profilepic from tbl_incidentreport as i left join tbl_incidentreport_detail as d on i.incidentreport_id = d.incidentreport_id left join tbl_adminuser as u on u.user_id = d.sender_id left join tbl_adminuser as a on a.user_id = d.receiver_id where d.incidentreport_id = ?");
        $all_conversations->bind_param("i",$incidentid);
        $result = $all_conversations->execute();
        $response = $all_conversations->get_result();
        $all_conversations->close();
        return $response;
    }

    public function contact_administrator($user_id){
        $all_conversations = $this->conn->prepare("select d.*,u.firstname as sendername,a.firstname as receivername,u.profilepic from tbl_conversations_contact_administrator as d left join tbl_adminuser as u on u.user_id = d.sender_id left join tbl_adminuser as a on a.user_id = d.receiver_id where d.sender_id = ? OR d.receiver_id = ?");
        $all_conversations->bind_param("ii",$user_id,$user_id);   
        $result = $all_conversations->execute();
        $response = $all_conversations->get_result();
        $all_conversations->close();
        return $response;
    }

    public function store_contract_administrator($user_id,$sendermessage){
        $receiver_id = 6;
        $send_by = 0; // customer
        $stmt = $this->conn->prepare("INSERT INTO tbl_conversations_contact_administrator (sender_id,receiver_id,message,send_by) VALUES (?,?,?,?)");
        $stmt->bind_param("iisi",$user_id,$receiver_id,$sendermessage,$send_by);
        if($stmt->execute()){
            return 1;
        }else{
            return 0;
        }
    }

    public function change_welcome_status($user_id,$welcome_status){
        $stmt = $this->conn->prepare("UPDATE tbl_customer SET welcome_status = ? WHERE user_id = ?");
        $stmt->bind_param("ii",$welcome_status,$user_id);
        if($stmt->execute()){
            return 1;
        }else{
            return 0;
        }
    }

    public function default_payment(){
        $stmt = $this->conn->prepare("select p.*,b.*,v.* from tbl_bank_payment as p join tbl_bankaccountdetail as b join tbl_vehicle as v where b.bankaccountdetail_id = p.bankaccountdetail_id  and v.vehicle_id = b.vehicle_id and p.payment_status='2'");
        $result = $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }

    public function get_all_notes($bank_payment_id){
        $stmt = $this->conn->prepare("SELECT n.*,a.firstname,a.lastname FROM `tbl_default_report_details` as n left join tbl_adminuser as a on a.user_id = n.user_id where n.bank_payment_id = ?");
        $stmt->bind_param("i",$bank_payment_id);
        $result = $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }


    public function get_requestpurchase_conversations($requestpurchase_id){
        $all_conversations = $this->conn->prepare("select d.*,u.firstname as sendername,a.firstname as receivername,u.profilepic from tbl_requestpurchase as i left join tbl_requestpurchase_details as d on i.requestpurchase_id = d.requestpurchase_id left join tbl_adminuser as u on u.user_id = d.sender_id left join tbl_adminuser as a on a.user_id = d.receiver_id where d.requestpurchase_id = ?");
        $all_conversations->bind_param("i",$requestpurchase_id);
        $result = $all_conversations->execute();
        $response = $all_conversations->get_result();
        $all_conversations->close();
        return $response;

    }

    public function insert_incident_details($incidentmsg,$incidentid,$user_id){
        $current_date = date("Y-m-d H:m:s");
        $receiver_id = 6;
        $stmt = $this->conn->prepare("INSERT INTO tbl_incidentreport_detail (incidentreport_id,sender_id,receiver_id,incident_detail,incidentreport_datetime) VALUES (?,?,?,?,?)");
        $stmt->bind_param("iiiss",$incidentid,$user_id,$receiver_id,$incidentmsg,$current_date);
        if($stmt->execute()){
            return 1;
        }else{
            return 0;
        }
    }

    public function insert_enquery_details($mssage,$senderid,$user_id,$inquiryid){
        $send_by = 0; // site manager
        $stmt = $this->conn->prepare("INSERT INTO tbl_inquirydetails (inquiry_id,sender_id,receiver_id,message,send_by) VALUES (?,?,?,?,?)");
        $stmt->bind_param("iiisi",$inquiryid,$user_id,$senderid,$mssage,$send_by);
        if($stmt->execute()){
            return 1;
        }else{
            return 0;
        }
    }



    public function store_proposed_purchase_msg($user_id,$message,$requestpurchase_id){
        $current_date = date("Y-m-d H:m:s");
        $receiver_id = 6; // its a admin id
        $sendby = 0;
        $stmt = $this->conn->prepare("INSERT INTO tbl_requestpurchase_details (requestpurchase_id,message,sender_id,receiver_id,sendby,msg_sent_datetime) VALUES (?,?,?,?,?,?)");
        $stmt->bind_param("isiiis",$requestpurchase_id,$message,$user_id,$receiver_id,$sendby,$current_date);   
        if($stmt->execute()){
            return 1;
        }else{
            return 0;
        }
    }

    
    

    /** Opal
     *  Generating random Unique MD5 String for user Api key 
     */
    public function generateApiKey($emi) {
        return md5(uniqid(rand(), true));
    }
	/** Opal 
	* Update the Api Key in customer table
	**/
	public function updateApikey($emailaddress = '', $api_key){
		if($emailaddress != ""){
			$stmt = $this->conn->prepare("UPDATE tbl_adminuser set api_key = ? WHERE emailaddress = ?");
			$stmt->bind_param("ss", $api_key, $emailaddress);
			$result = $stmt->execute();
			$stmt->close();
		}else{
			//echo $api_key;
			$key = NULL;
			$stmt = $this->conn->prepare("UPDATE tbl_adminuser SET api_key = ? WHERE api_key = ?");
			$stmt->bind_param("is", $key, $api_key);			
			$result = $stmt->execute();
			$stmt->close();
			
		}
		return $result;
       
	}
    /** Opal
     * Fetching user id by api key
     * @param String $api_key user api key
     */
    public function getUserId($api_key) {
        $stmt = $this->conn->prepare("SELECT user_id FROM tbl_adminuser WHERE api_key = ?");
        $stmt->bind_param("s", $api_key);
        if ($stmt->execute()) {
            $stmt->bind_result($user_id);
            $stmt->fetch();
            // TODO
            // $user_id = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            return $user_id;
        } else {
            return NULL;
        }
    }


    public function getTeam(){
        $stmt = $this->conn->prepare("SELECT * FROM `tbl_adminuser` as a, tbl_usertype as u where a.usertype_id = u.usertype_id AND a.status =1 ");
        $result = $stmt->execute();
        if (false === $result) {
            die('execute() failed: ' . htmlspecialchars($stmt->error));
        }
        $team_data = $stmt->get_result();
        $stmt->close();
        return $team_data;
    }
    /**
     * Fetching all user tasks
     * @param String $user_id id of the user
    */
    public function getWarningLights() {
        $stmt = $this->conn->prepare("SELECT * FROM tbl_cms where warning_lights = '1' AND status = '1'");
        $stmt->execute();
        $warningLights = $stmt->get_result();
        $stmt->close();
        return $warningLights;
    }
    /* Here we fetch all jobs for the technition */
    public function fetchAlljobs($user_id){
		
        $stmt = $this->conn->prepare("SELECT a.scheduledatetime,c.plateno, c.model, d.name, a.job_id FROM tbl_job a left join tbl_booking b on a.booking_id = b.booking_id join tbl_vehicle c on c.vehicle_id = b.vehicle_id join tbl_servicetype d on d.servicetype_id = a.service_id WHERE a.technician_id = ? AND (a.status = 0 OR a.status = 1 OR a.status = 2)"); 
        $stmt->bind_param("i",$user_id);
        $result = $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }
    /* End */

    public function get_job_applications(){
        $stmt = $this->conn->prepare("SELECT u.*,ut.name as rolename FROM tbl_adminuser as u left join tbl_usertype as ut on ut.usertype_id = u.usertype_id WHERE u.status = 2");
        $result = $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }


    public function getSalesmanData($id){
		$stmt = $this->conn->prepare("SELECT * FROM tbl_adminuser as a, tbl_bankaccountdetail as b WHERE a.user_id = b.user_id AND b.user_id = ? AND b.vehicle_id = 0  AND a.status = 2");
		$stmt->bind_param("i", $id);
		$stmt->execute();
		$return_arrray['error'] = false;
		//$return_arrray['jobApplicationData'] = array();
		$pending_application = $stmt->get_result();
		while($row_pending_application = $pending_application->fetch_assoc()){
			$account_no  = explode("-", $row_pending_application["accountnumber"]);
			$return_arrray["user_id"] = $row_pending_application["user_id"];
			$return_arrray["first_name"] = $row_pending_application["firstname"];
			$return_arrray["last_name"] = $row_pending_application["lastname"];
			$return_arrray["date_of_birth"] = $row_pending_application["dateofborth"];
			$return_arrray["gender"] = $row_pending_application["gender"];
			$return_arrray["mobile_no"] = $row_pending_application["mobilenumber"];
			$return_arrray["address"] = $row_pending_application["address"];
			$return_arrray["address2"] = $row_pending_application["address2"];
			$return_arrray["email_address"] = $row_pending_application["emailaddress"];
			$return_arrray["driver_licence_no"] = $row_pending_application["drivinglicencenumber"];
			$return_arrray["driver_licence_expiry_date"] = $row_pending_application["drivinglicenceexpiredate"];
			$return_arrray["bank_account_name"] = $row_pending_application["accountname"];
			$return_arrray["bank_name"] = $row_pending_application["bankname"];
			$return_arrray["bank_account_no1"] = $account_no[0];
			$return_arrray["bank_account_no2"] = $account_no[1];
			$return_arrray["bank_account_no3"] = $account_no[2];
			$return_arrray["bank_account_no4"] = $account_no[3];
			$return_arrray["payment_frequency"] = $row_pending_application["paymentfrequency"];
			$return_arrray["payment_day"] = $row_pending_application["paymentday"];
			$return_arrray["nzlProof1"] = $row_pending_application["drivinglicencefile1"];
			$return_arrray["nzlProof2"] = $row_pending_application["drivinglicencefile2"];
			$return_arrray["profilepic"] = $row_pending_application["profilepic"];
            $return_arrray["contract"] = $row_pending_application["contract"];
            $return_arrray["hourly_rate"] = $row_pending_application["hourly_rate"];
		}
		return $return_arrray;
    }

    public function getLastTwoService($vehicle_id){
        //echo "SELECT b.bookingdatetime, b.bookingendtime, jc.oilfilter_number, jc.airfilter_number, jc.odometerreading, jc.notes FROM `tbl_job` as j, tbl_jobcard as jc, tbl_booking as b, tbl_vehicle as v WHERE b.vehicle_id = v.vehicle_id AND j.booking_id = b.booking_id AND jc.booking_id=b.booking_id and v.vehicle_id =".$vehicle_id." GROUP BY b.booking_id LIMIT 1,2";
            $stmt = $this->conn->prepare("SELECT b.bookingdatetime, b.bookingendtime, jc.oilfilter_number, jc.airfilter_number, jc.odometerreading, jc.notes, b.servicetype_id, jc.engineoil_amount FROM tbl_jobcard as jc, tbl_booking as b, tbl_vehicle as v WHERE b.vehicle_id = v.vehicle_id AND jc.booking_id=b.booking_id and v.vehicle_id = ? AND b.job_status = 4 GROUP BY b.booking_id LIMIT 0,2");
            $stmt->bind_param("i", $vehicle_id);
            $stmt->execute();
            $bank_data = $stmt->get_result();
            $return_arrray=array();
            while ($bank = $bank_data->fetch_assoc()) {
                $tmp = array();
                $date = new DateTime($bank['bookingendtime']);
                $match_database_date = $date->format('Y-m-d');
                $tmp['booking_date'] = $match_database_date;
                $tmp['oilfilter_number'] = $bank['oilfilter_number'];
                $tmp['airfilter_number'] = $bank['airfilter_number'];
                $tmp['odometerreading'] = $bank['odometerreading'];
                $tmp['notes'] = $bank['notes'];
                $tmp['engineoil_amount'] = $bank['engineoil_amount'];
                $service_type_query =  $this->conn->prepare("SELECT name FROM tbl_servicetype WHERE servicetype_id = ? "); 
                $service_type_query->bind_param("i",$bank['servicetype_id']);
                $service_type_query->execute();
                $service_type_query_result = $service_type_query->get_result();
                $service_type_query_data  = $service_type_query_result->fetch_assoc();
                $service_name = $service_type_query_data['name'];
                $tmp['service_type'] = $service_name;
                array_push($return_arrray, $tmp);
            }
            //print_r($return_arrray);die();
            return $return_arrray;
        }
    

    public function remove_public_holidays_user($publicholidaysid,$user_id){
        $stmt = $this->conn->prepare("DELETE FROM tbl_publicholidays WHERE id = ? AND site_manager_id = ?");
        $stmt->bind_param("ii",$publicholidaysid,$user_id);
        if($stmt->execute()){
            return 1;
        }else{
            return 2;
        }
    }

    /* get all team members */
    public function getallmembers($user_id){
        $stmt = $this->conn->prepare("select u.*,t.name as rolename from tbl_usertype as t right join tbl_adminuser as u on t.usertype_id = u.usertype_id where u.addedbyuser_id = ?");
        $stmt->bind_param("i",$user_id);
        $result = $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }
    /* End */

    


    

    public function get_my_members($user_id){
        $stmt = $this->conn->prepare("select u.*,t.name as rolename from tbl_usertype as t right join tbl_adminuser as u on t.usertype_id = u.usertype_id where u.addedbyuser_id = ?");
        $stmt->bind_param("i",$user_id);
        $result = $stmt->execute();
        $response = $stmt->get_result();
        $response_data = array();
        while ($team_members = $response->fetch_assoc()) {
            $tmp = array();
            $user_id = $team_members["user_id"];
            $tmp["registration_date"] = date('Y-m-d',strtotime($team_members["registration_date"]));
            $tmp["name"] = $team_members["firstname"];
            $tmp["roles"] = $team_members["rolename"];
            $tmp["phone"] = $team_members["mobilenumber"];
            $tmp["userid"] = $team_members["user_id"];
            array_push($response_data, $tmp);
        }
        return $response_data;
        
    }

    

    public function getall_past_members($user_id){
        $stmt = $this->conn->prepare("select u.*,t.name as rolename from tbl_usertype as t right join tbl_adminuser as u on t.usertype_id = u.usertype_id where u.addedbyuser_id = ? AND u.registrationend_date IS NOT NULL");
        $stmt->bind_param("i",$user_id);
        $result = $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }

    public function get_user_info($userid){
        $query = $this->conn->prepare("select u.*,t.name as rolename  from tbl_usertype as t right join tbl_adminuser as u on t.usertype_id = u.usertype_id  where u.user_id = ? ");
        $query->bind_param("i",$userid);
        $result = $query->execute();
        $response = $query->get_result();
        $query->close();
        return $response;
    }

    public function get_customer_info($userid){
        $query = $this->conn->prepare("select * from tbl_customer where user_id = ?");
        $query->bind_param("i",$userid);
        $result = $query->execute();
        $response = $query->get_result();
        $query->close();
        return $response;

    }

    public function user_past_info($userid){
        $query = $this->conn->prepare("SELECT * FROM tbl_contactdetails WHERE user_id = ? and user_type = 1");
        $query->bind_param("i",$userid);
        $result = $query->execute();
        $data = $query->get_result();
        $response = array();
        while($past_details = $data->fetch_assoc()){
            $temp_data = array();
            $temp_data['date'] = date("d-m-Y H:m:i",strtotime($past_details['updated_datetime']));
            $temp_data['address1'] = $past_details['address1'];
            $temp_data['address2'] = $past_details['address2'];
            array_push($response,$temp_data);
        }
        $query->close();
        return $response;
    }

    public function get_avg_site_productivity(){
        $stmt = $this->conn->prepare("SELECT avg(total_time) as site_average FROM `tbl_booking` where job_status = 4");
        $stmt->execute();
        $data = $stmt->get_result();
        $site_avg_data = $data->fetch_assoc();
        return $site_avg_data['site_average'];
    }

    /**
     * Fetching single task
     * @param String $task_id id of the task
     */
    public function getWarningLight($cms_id) {
        $stmt = $this->conn->prepare("SELECT cms_id, cms_title, cms_description, cms_icon from tbl_cms WHERE cms_id = ?");
        $stmt->bind_param("i", $cms_id);
        if ($stmt->execute()) {
            $res = array();
            $stmt->bind_result($cms_id, $cms_title, $cms_description, $cms_icon);
            // TODO
            // $task = $stmt->get_result()->fetch_assoc();
            $stmt->fetch();
            $res["cms_id"] = $cms_id;
            $res["cms_title"] = $cms_title;
            $res["cms_description"] = $cms_description;
            $res["cms_icon"] = $cms_icon;
            $stmt->close();
            return $res;
        } else {
            return NULL;
        }
    }

    public function getallvehicle(){
        $stmt = $this->conn->prepare("select * from tbl_vehicle order by vehicle_id DESC");
        $stmt->execute();
        $all_vehicle_data = $stmt->get_result();
        $stmt->close();
        return $all_vehicle_data;
    }
    public function get_car_data($serachData){
		$param = "%{$serachData}%";
        $stmt = $this->conn->prepare("select * from tbl_vehicle where plateno LIKE ? OR make LIKE ? OR model LIKE ? order by vehicle_id DESC");
        $stmt->bind_param("sss",$param,$param,$param);
        $stmt->execute();
        $all_vehicle_data = $stmt->get_result();
        $stmt->close();
        return $all_vehicle_data;
    }
    public function store_technician_hours($user_id,$reportdate,$techniciandata){
        $reportdate  = date("Y-m-d",strtotime($reportdate));
        foreach($techniciandata as $technician){
            $stmt = $this->conn->prepare("INSERT INTO tbl_technicianhours (sitemanager_id,technician_id,start_time,finish_time,reportDate,total_time) VALUES (?,?,?,?,?,?)");
            $stmt->bind_param("iissss",$user_id,$technician['user_id'],$technician['starttime'],$technician['endtime'],$reportdate,$technician['hours']);
            $stmt->execute();
            $stmt->close();
        }
        return 1;
    }
    public function get_my_vehicles($userid){
        $stmt = $this->conn->prepare("SELECT v.*,b.booking_id FROM tbl_vehicle as v left join tbl_booking as b on v.vehicle_id = b.vehicle_id WHERE v.user_id = ?");
        $stmt->bind_param("i",$userid);
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }
    public function get_vehicle_booking_info($booking_id){
        $stmt = $this->conn->prepare("SELECT b.*,s.name as servicetypename FROM `tbl_booking` as b left join tbl_servicetype as s on s.servicetype_id = b.servicetype_id WHERE b.booking_id = ?");
        $stmt->bind_param("i",$booking_id);
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;


    }

    public function getTotalService($vehicle_id){
		$stmt = $this->conn->prepare("SELECT count(*) as b FROM `tbl_booking` WHERE vehicle_id = ?");
        $stmt->bind_param("i", $vehicle_id);
		$result = $stmt->execute();
        $cars_data = $stmt->get_result();
		while ($cars =  $cars_data->fetch_assoc()) {
			$total_number = $cars['b'];
		}

		return $total_number;
    }
    
    public function getServiceId($plan_id, $getTotalService) {
		$stmt = $this->conn->prepare("SELECT servicetype_id FROM tbl_plandetails WHERE plan_id = ? ORDER BY plandetail_id ASC LIMIT ?,1");
        $stmt->bind_param("ii", $plan_id, $getTotalService);
        if ($stmt->execute()) {
			
            $stmt->bind_result($servicetype_id);
            $stmt->fetch();

			//ECHO $service_id;DIE();
            // TODO
            // $user_id = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            return $servicetype_id;
        } else {
            return NULL;
        }
    }

    public function GetServiceDuration($service_id) {
		$stmt = $this->conn->prepare("SELECT duration FROM tbl_servicetype WHERE servicetype_id = ?");
        $stmt->bind_param("i", $service_id);
        if ($stmt->execute()) {
            $stmt->bind_result($duration);
            $stmt->fetch();
            $stmt->close();
            return $duration;
        } else {
            return NULL;
        }
    }
    public function insertBooking($service_id, $user_id, $vehicle_id, $booking_time, $location_address, $pickup_time, $dropoff_time, $duration, $booking_end_time){
		$stmt = $this->conn->prepare("INSERT INTO tbl_booking (vehicle_id, user_id, servicetype_id, location, carpickuptime, cardropofftime, bookingdatetime, bookingendtime, duration) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
		//echo "".$user_id."=".$lat."=".$lan."=".$datetime."=".$api_key;
		$stmt->bind_param("iiissssss", $vehicle_id, $user_id, $service_id, $location_address, $pickup_time, $dropoff_time, $booking_time, $booking_end_time, $duration );
        $result = $stmt->execute();
		$booking_id = $stmt->insert_id;
        $stmt->close();



		//Insert in to jobcard
		$stmt = $this->conn->prepare("INSERT INTO tbl_jobcard (booking_id) VALUES (?)");
		//echo "".$user_id."=".$lat."=".$lan."=".$datetime."=".$api_key;
		$stmt->bind_param("i", $booking_id);
        $result = $stmt->execute();
		$job_card_id = $stmt->insert_id;
        $stmt->close();

		$allservices_query = $this->conn->prepare("SELECT * FROM tbl_servicechecklist WHERE servicetype_id = ? AND parent_id = 0 "); 
        $allservices_query->bind_param("i",$service_id);
        $allservices_query->execute();
        $allservices_query_result = $allservices_query->get_result();
        while($allservices_query_data = $allservices_query_result->fetch_assoc()){
            $allservices[]  = $allservices_query_data;
        }
        if(! empty($allservices)){
            foreach($allservices as $servicetype){
                $parent_id = $servicetype['parent_id'];
                if($parent_id == 0){
						
					//Insert in to jobcard
					$stmt1 = $this->conn->prepare("INSERT INTO tbl_jobcard_detail (jobcard_id, service_type_name, parent_id) VALUES (?,?,?)");
					$stmt1->bind_param("isi", $job_card_id, $servicetype['name'], $parent_id);
					$result = $stmt1->execute();
					$job_card_detail_id = $stmt1->insert_id;
					$stmt1->close();
						
                    $servicechecklist_id = $servicetype['servicechecklist_id'];
                    // here we fetch all sub category
                    $allsubservice_query = $this->conn->prepare("SELECT *, GROUP_CONCAT(checklist_value) as options FROM tbl_servicechecklist as s, tbl_servicechecklistdetail as sd WHERE s.servicechecklist_id=sd.servicechecklist_id AND parent_id = ? GROUP BY sd.servicechecklist_id "); 
                    $allsubservice_query->bind_param("i",$servicechecklist_id);
                    $allsubservice_query->execute();
                    $allsubservice_result = $allsubservice_query->get_result();
					$l = 0;
                    while($allsubservice_data  = $allsubservice_result->fetch_assoc()){
                        //Insert in to jobcard_Detail
						$stmt2 = $this->conn->prepare("INSERT INTO tbl_jobcard_detail (jobcard_id, service_type_name, parent_id, service_type_options) VALUES (?,?,?,?)");
						$stmt2->bind_param("isis", $job_card_id, $allsubservice_data['name'], $job_card_detail_id, $allsubservice_data['options']);
						$result = $stmt2->execute();
						$stmt2->close();
                            
                    }   
                    // End 
                }
            }
        }   

        if ($result) {
            return true;
        } else {
            return NULL;
        }
    }
    


    public function get_vehicle_detail_data($vehicle_id){
        $stmt = $this->conn->prepare("select v.*,vd.start_date as signupdate,vd.plan_id,vd.end_date as renewaldate,c.name as categoryname,e.name as enginetypename,p.planname as type,p.duration as term,p.special_benefits from tbl_vehicle as v left join tbl_vehicleplandetail as vd ON v.vehicle_id = vd.vehicle_id left join tbl_category as c on c.category_id = v.category_id left join tbl_enginetype as e on e.enginetype_id = v.enginetype_id left join tbl_plan as p on p.plan_id = v.plan_id where v.vehicle_id = ?");
        $stmt->bind_param("i",$vehicle_id);
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }

    public function get_vehicle_id($plateno){
        $stmt =  $this->conn->prepare("select vehicle_id from tbl_vehicle where plateno = ?");
        $stmt->bind_param("s",$plateno);
        $stmt->execute();
        $data = $stmt->get_result();
        $vehicle_info = $data->fetch_assoc();
        $vehicle_id = $vehicle_info['vehicle_id'];
        return $vehicle_id;
    }

    public function user_contract_details($user_id){
        $stmt = $this->conn->prepare("select c.*,v.plateno,cat.name as categoryname,e.name as enginetypename,p.planname,p.duration,p.special_benefits from tbl_customer as c left join tbl_vehicle as v on c.user_id = v.user_id left join tbl_category as cat on cat.category_id = v.category_id left join tbl_enginetype as e on e.enginetype_id = v.enginetype_id left join tbl_plan as p on p.plan_id = v.plan_id where c.user_id = ?");
        $stmt->bind_param("i",$user_id);      
        $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }

    public function getTotalServiceApplayonvehicle($vehicle_id){
        $stmt = $this->conn->prepare("SELECT b.bookingdatetime,c.firstname,jc.jobcard_pdf from tbl_booking as b left join tbl_customer as c on c.user_id = b.user_id left join tbl_jobcard as jc on b.booking_id = jc.booking_id where b.vehicle_id = ?");
        $stmt->bind_param("i",$vehicle_id); 
        $stmt->execute();
        $data = $stmt->get_result();
        $response = array();
        while($booking_data = $data->fetch_assoc()){
            $temp_data = array();
            $temp_data['date'] = date("d-m-Y",strtotime($booking_data['bookingdatetime']));
            $temp_data['time'] = date("H:m:s",strtotime($booking_data['bookingdatetime']));
            $temp_data['firstname'] = $booking_data['firstname'];
            $temp_data['pdfname'] = $booking_data['jobcard_pdf'];
            array_push($response,$temp_data);
        }
        $stmt->close();
        return $response;
    }

    public function assign_job_technitions($booking_id,$technician_id){
        $status = 0;

        // get booking date time 
        $stmt = $this->conn->prepare("SELECT * FROM tbl_booking WHERE booking_id = ?");
        $stmt->bind_param("i",$booking_id);
        $stmt->execute();
        $booking_data = $stmt->get_result();
        $bank = $booking_data->fetch_assoc();
        $scheduledatetime = $bank['bookingdatetime'];
        // end 
        
        $update_query = $this->conn->prepare("UPDATE tbl_booking SET technician_id = ?,scheduledatetime = ?,job_status = ? WHERE booking_id = ?");
        $update_query->bind_param("isii",$technician_id,$scheduledatetime,$status,$booking_id);
        if($update_query->execute()){
            return 1;
        }else{
            return 0;
        }

    }

    public function get_job_in_progress(){
        $stmt = $this->conn->prepare("select b.scheduledatetime,b.total_time,v.plateno,s.name as servicename,au.firstname as technitionname from tbl_booking as b left join tbl_vehicle as v On v.vehicle_id = b.vehicle_id left join tbl_servicetype as s on s.servicetype_id = b.servicetype_id left join tbl_customer as c on c.user_id = v.user_id left join tbl_adminuser as au on au.user_id = c.addedbyuser_id where b.job_status = '1' OR b.job_status = '2'");
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }

    public function get_all_technitions($user_id){
        $stmt = $this->conn->prepare("SELECT * FROM `tbl_adminuser` where addedbyuser_id = ? AND usertype_id = 4");
        $stmt->bind_param("i",$user_id);
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }

    public function get_technitions_completed_jobs($user_id){
        $stmt = $this->conn->prepare("SELECT u.* , (select count(*) from tbl_booking where technician_id = u.user_id AND (job_status = 3 OR job_status = 4) AND date(bookingendtime) = CURDATE()) as total_service_completed FROM `tbl_adminuser` as u where u.addedbyuser_id = ? AND u.usertype_id = 4");
        $stmt->bind_param("i",$user_id);    
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }


    public function get_all_historicaljob(){
        $stmt = $this->conn->prepare("SELECT b.scheduledatetime as startdate,v.plateno,a.firstname,jc.jobcard_pdf as pdfname FROM `tbl_booking` as b left join tbl_vehicle as v on v.vehicle_id = b.vehicle_id left join tbl_customer as c on c.user_id = b.user_id  left join tbl_adminuser as a on a.user_id = c.addedbyuser_id left join tbl_jobcard as jc on b.booking_id = jc.booking_id where b.job_status = 4");  
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }

    public function get_all_completedjob(){
        $stmt = $this->conn->prepare("SELECT b.scheduledatetime as startdate,b.booking_id,v.plateno,c.firstname as name ,ad.firstname as technicianname FROM `tbl_booking` as b left join tbl_vehicle as v on v.vehicle_id = b.vehicle_id left join tbl_adminuser as ad on ad.user_id = b.technician_id left join tbl_customer as c on c.user_id = b.user_id where b.job_status = 3"); 
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }

    public function get_all_not_assign_job(){
        $stmt = $this->conn->prepare("SELECT b.booking_id,b.vehicle_id,v.plateno,c.firstname as name FROM `tbl_booking` as b left join tbl_vehicle as v on v.vehicle_id = b.vehicle_id left join tbl_customer as c on c.user_id = b.user_id where b.job_status IS NULL"); 
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }
    public function get_service_history($customer_id){
        $stmt = $this->conn->prepare("SELECT b.bookingdatetime,j.jobcard_pdf as pdfname,v.make,v.model,v.plateno FROM `tbl_booking` as b left join tbl_jobcard as j on b.booking_id = j.booking_id LEFT join tbl_vehicle as v on v.vehicle_id = b.vehicle_id WHERE b.user_id = ?");
        $stmt->bind_param("i",$customer_id);
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }

    public function getLastService($vehicle_id){
        //echo "SELECT b.bookingdatetime, b.bookingendtime, jc.oilfilter_number, jc.airfilter_number, jc.odometerreading, jc.notes FROM `tbl_job` as j, tbl_jobcard as jc, tbl_booking as b, tbl_vehicle as v WHERE b.vehicle_id = v.vehicle_id AND j.booking_id = b.booking_id AND jc.booking_id=b.booking_id and v.vehicle_id =".$vehicle_id." GROUP BY b.booking_id LIMIT 1,2";
            $stmt = $this->conn->prepare("SELECT b.bookingdatetime, b.bookingendtime, jc.oilfilter_number, jc.airfilter_number, jc.odometerreading, jc.notes FROM tbl_jobcard as jc, tbl_booking as b, tbl_vehicle as v WHERE b.vehicle_id = v.vehicle_id AND jc.booking_id=b.booking_id and v.vehicle_id =".$vehicle_id." GROUP BY b.booking_id LIMIT 1");
            $stmt->execute();
            $bank_data = $stmt->get_result();
            $return_arrray=array();

            $bank = $bank_data->fetch_assoc();
            $tmp['notes'] = $bank['notes'];
            $date = new DateTime($bank['bookingendtime']);
            $match_database_date = $date->format('Y-m-d');
            $tmp['booking_date'] = $match_database_date;
            $tmp['oilfilter_number'] = $bank['oilfilter_number'];
            $tmp['airfilter_number'] = $bank['airfilter_number'];
            $tmp['odometerreading'] = $bank['odometerreading'];
            
            $service_type_query =  $this->conn->prepare("SELECT name FROM tbl_servicetype WHERE servicetype_id = ? "); 
            $service_type_query->bind_param("i",$bank['servicetype_id']);
            $service_type_query->execute();
            $service_type_query_result = $service_type_query->get_result();
            $service_type_query_data  = $service_type_query_result->fetch_assoc();
            $service_name = $service_type_query_data['name'];
            $tmp['service_type'] = $service_name;
            return $tmp['booking_date'];    
            

            // while () {
            //     $tmp = array();
            //     $date = new DateTime($bank['bookingendtime']);
            //     $match_database_date = $date->format('Y-m-d');
            //     $tmp['booking_date'] = $match_database_date;
            //     $tmp['oilfilter_number'] = $bank['oilfilter_number'];
            //     $tmp['airfilter_number'] = $bank['airfilter_number'];
            //     $tmp['odometerreading'] = $bank['odometerreading'];
                
            //     $service_type_query =  $this->conn->prepare("SELECT name FROM tbl_servicetype WHERE servicetype_id = ? "); 
            //     $service_type_query->bind_param("i",$bank['servicetype_id']);
            //     $service_type_query->execute();
            //     $service_type_query_result = $service_type_query->get_result();
            //     $service_type_query_data  = $service_type_query_result->fetch_assoc();
            //     $service_name = $service_type_query_data['name'];
            //     $tmp['service_type'] = $service_name;
            //     array_push($return_arrray, $tmp);
            // }
            //print_r($return_arrray);die();
            
    }


    public function saveTechnician($user_id,$type,$first_name,$last_name,$date_of_birth,$gender,$mobile_no,$address,$email_address,$driver_licence_no,$driver_licence_expiry_date,$driving_licence_proof,$profile_pic_name,$bank_account_name,$bank_account_no,$bank_name,$payment_frequency,$payment_day,$hourly_rate,$technicians_signature,$status,$declarationcontent,$signature_proof,$technician_id){
        $registration_date = date("Y-m-d H:i:s");
        $password = base64_encode("123456");
        $date_of_birth = date("Y-m-d",strtotime($date_of_birth));
        $driver_licence_expiry_date = date("Y-m-d",strtotime($driver_licence_expiry_date));

        if(isset($technician_id) && $technician_id != ''){ // Update the information 
            $update_query = $this->conn->prepare("UPDATE tbl_adminuser SET firstname = ?,lastname = ?,emailaddress = ?,dateofborth = ?,mobilenumber = ?,address = ?,drivinglicencenumber = ?,drivinglicenceexpiredate = ?,drivinglicencefile1 = ?,profilepic = ?,declarationcontent = ?,gender = ?,status = ?,signature_proof = ? WHERE user_id = ?");
            $update_query->bind_param("ssssssssssssssi",$first_name,$last_name,$email_address,$date_of_birth,$mobile_no,$address,$driver_licence_no,$driver_licence_expiry_date,$driving_licence_proof,$profile_pic_name,$declarationcontent,$gender,$status,$signature_proof,$technician_id);
            if($update_query->execute()){
                //  Here we also update the bank details information
                $update_bank_details = $this->conn->prepare("UPDATE tbl_bankaccountdetail SET accountname = ?,accountnumber = ?,bankname = ?,hourly_rate = ? WHERE user_id = ?");
                $update_bank_details->bind_param('ssssi',$bank_account_name,$bank_account_no,$bank_name,$hourly_rate,$technician_id);
                $update_bank_details->execute();
                $update_bank_details->close();
                // End 
                return 3;
            }else{
                return 4;
            }
        }else{
            $save_query = $this->conn->prepare("INSERT INTO tbl_adminuser (addedbyuser_id,usertype_id,firstname,lastname,emailaddress,password,dateofborth,mobilenumber,address,drivinglicencenumber,drivinglicenceexpiredate,drivinglicencefile1,profilepic,declarationcontent,gender,status,registration_date,signature_proof) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
            $save_query->bind_param("iisssssssssssssiss",$user_id,$type,$first_name,$last_name,$email_address,$password,$date_of_birth,$mobile_no,$address,$driver_licence_no,$driver_licence_expiry_date,$driving_licence_proof,$profile_pic_name,$declarationcontent,$gender,$status,$registration_date,$signature_proof);
            if($save_query->execute()){
                $technician_id = $this->conn->insert_id;
                // other information will store in bank account table
                $bank_info_query = $this->conn->prepare("INSERT INTO tbl_bankaccountdetail (user_id,accountname,accountnumber,bankname,paymentfrequency,paymentday,hourly_rate) VALUES (?,?,?,?,?,?,?)");
                $bank_info_query->bind_param('issssss',$technician_id,$bank_account_name,$bank_account_no,$bank_name,$payment_frequency,$payment_day,$hourly_rate);
                $bank_info_query->execute();
                $bank_info_query->close();
                $save_query->close();
                // return 1;
                return $technician_id;
            }else{
                $save_query->close();
                // return 2;
                return "error";
            }
        }


        
    }
    public function getUserinfo($email_address){
        $stmt = $this->conn->prepare("SELECT user_id FROM tbl_adminuser WHERE emailaddress = ? ");
        $stmt->bind_param("s",$email_address);
        $result = $stmt->execute();
        $mybookingdata = $stmt->get_result();
        $data = $mybookingdata->fetch_assoc();
        $user_id = $data['user_id'];
        return $user_id;
    }

    public function getAllFeedback($user_id){
        $stmt = $this->conn->prepare("select u.firstname,u.user_id,c.conversation_datetime,c.conversation_id from tbl_adminuser as u right join tbl_conversation as c on u.user_id = c.sender_id where (c.sender_id = ? OR c.receiver_id = ?) AND c.status = 0");
        $stmt->bind_param("is",$user_id,$user_id);
        $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }

    public function get_all_archivefeedback($user_id){
        $stmt = $this->conn->prepare("select u.firstname,u.user_id,c.conversation_datetime,c.conversation_id,c.pdf_filename from tbl_adminuser as u right join tbl_conversation as c on u.user_id = c.sender_id where (c.sender_id = ? OR c.receiver_id = ?) AND c.status = 1");
        $stmt->bind_param("is",$user_id,$user_id);
        $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }
    public function datewise_service_completed($my_date,$service_id){
        $stmt = $this->conn->prepare("select count(*) as total_service_completed from tbl_booking where servicetype_id = ? AND job_status = 4 AND date(bookingendtime) = ?");
        // select count(*) as completed_service from tbl_booking where job_status = 4 AND date(bookingendtime) = ?
        $stmt->bind_param("is",$service_id,$my_date);
        $stmt->execute();
        $date_count = $stmt->get_result();
        $my_count = $date_count->fetch_assoc();
        $stmt->close();
        return $my_count['total_service_completed'];
    }

    public function get_parts_reports_daily($my_date,$colname){
        $stmt = $this->conn->prepare("select COALESCE(SUM(j.engineoil_amount),0) as sum_engineoil ,COALESCE(SUM(j.breakfluid_amount),0) as sum_breakfluid , COALESCE(SUM(j.coolant_amount),0) as sum_coolant,count(oilfilter_number) as count_olifilter,count(airfilter_number) as count_airfilter ,COALESCE(SUM(j.clutchfluid_amount),0) as sum_clutchfluid from tbl_booking as b left join tbl_jobcard as j on b.booking_id = j.booking_id  where b.job_status = 4 AND date(b.bookingendtime) = ? ");
        $stmt->bind_param("s",$my_date);
        $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }

    public function get_engine_oil($my_date,$columname){
        $stmt = $this->conn->prepare("select COALESCE(SUM(j.".$columname."),0) as sum_engineoil from tbl_booking as b left join tbl_jobcard as j on b.booking_id = j.booking_id  where b.job_status = 4 AND date(b.bookingendtime) = ? ");
        $stmt->bind_param("s",$my_date);
        $stmt->execute();
        $date_count = $stmt->get_result();
        $my_count = $date_count->fetch_assoc();
        $stmt->close();
        return $my_count['sum_engineoil'];
    }

    public function get_count_data($my_date,$columname){
        $stmt = $this->conn->prepare("select count(j.".$columname.") as count_data from tbl_booking as b left join tbl_jobcard as j on b.booking_id = j.booking_id  where b.job_status = 4 AND date(b.bookingendtime) = ? ");
        $stmt->bind_param("s",$my_date);
        $stmt->execute();
        $date_count = $stmt->get_result();
        $my_count = $date_count->fetch_assoc();
        $stmt->close();
        return $my_count['count_data'];
    }

    public function weekly_service_completed($start_date,$end_date,$service_id){
        $stmt = $this->conn->prepare("select count(*) as total_service_completed from tbl_booking where servicetype_id = ? AND job_status = 4 AND date(bookingendtime) >= ? AND  date(bookingendtime) <= ?"); //BETWEEN ? AND ? 
        // select count(*) as completed_service from tbl_booking where job_status = 4 AND date(bookingendtime) = ?
        $stmt->bind_param("iss",$service_id,$start_date,$end_date);
        $stmt->execute();
        $date_count = $stmt->get_result();
        $my_count = $date_count->fetch_assoc();
        $stmt->close();
        return $my_count['total_service_completed'];
    }

    public function weekly_parts_reports($start_date,$end_date,$columname){
        $stmt = $this->conn->prepare("select COALESCE(SUM(j.".$columname."),0) as sum_engineoil from tbl_booking as b left join tbl_jobcard as j on b.booking_id = j.booking_id  where b.job_status = 4 AND date(bookingendtime) >= ? AND  date(bookingendtime) <= ?");
        $stmt->bind_param("ss",$start_date,$end_date);        
        $stmt->execute();
        $date_count = $stmt->get_result();
        $my_count = $date_count->fetch_assoc();
        $stmt->close();
        return $my_count['sum_engineoil'];
    }

    public function get_technitions_weekly_completed_services($start_date,$end_date,$technician_id){
        if(isset($technician_id) && $technician_id !=''){
            $stmt = $this->conn->prepare("select count(*) as total_service_completed from tbl_booking where job_status = 4 AND date(bookingendtime) >= ? AND  date(bookingendtime) <= ? AND technician_id = ?");
            $stmt->bind_param("ssi",$start_date,$end_date,$technician_id);        
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['total_service_completed'];
        }else{
            $stmt = $this->conn->prepare("select count(*) as total_service_completed from tbl_booking where job_status = 4 AND date(bookingendtime) >= ? AND  date(bookingendtime) <= ?");
            $stmt->bind_param("ss",$start_date,$end_date);        
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['total_service_completed'];
        }
        
    }

    public function get_weeklysite_time_avg($start_date,$end_date,$technician_id){
        $stmt = $this->conn->prepare("select COALESCE(avg(total_time),0) as avg_total_time from tbl_booking where job_status = 4 AND date(bookingendtime) >= ? AND  date(bookingendtime) <= ? AND technician_id = ? ");
        $stmt->bind_param("ssi",$start_date,$end_date,$technician_id);        
        $stmt->execute();
        $date_count = $stmt->get_result();
        $my_count = $date_count->fetch_assoc();
        $stmt->close();
        return $my_count['avg_total_time'];
    }

    public function get_weekly_technition_avg($start_date,$end_date,$technitions_id,$service_id){
        $stmt = $this->conn->prepare("select COALESCE(avg(total_time),0) as avg_total_time from tbl_booking where job_status = 4 AND date(bookingendtime) >= '".$start_date."' AND  date(bookingendtime) <= '".$end_date."' AND technician_id = ? AND servicetype_id = ?");
        $stmt->bind_param("ii",$technitions_id,$service_id);        
        $stmt->execute();
        $date_count = $stmt->get_result();
        $my_count = $date_count->fetch_assoc();
        $stmt->close();
        return $my_count['avg_total_time'];

    }

    public function get_salesmandaily_report($my_date,$salesman_id){
        if(isset($salesman_id) && $salesman_id !=''){
            $get_saleman_vehicle = $this->conn->prepare("select v.vehicle_id from tbl_vehicle as v LEFT join tbl_vehiclesales_detail as d on v.vehicle_id = d.vehicle_id where v.user_id = ?  AND date(datetime) = '".$my_date."' GROUP by d.vehicle_id");
            $get_saleman_vehicle->bind_param("i",$salesman_id);
            $get_saleman_vehicle->execute();
            $vehicle = $get_saleman_vehicle->get_result();
            $get_saleman_vehicle->close();
            return $vehicle->num_rows;
        }else{
            $get_saleman_vehicle = $this->conn->prepare("select v.vehicle_id from tbl_vehicle as v LEFT join tbl_vehiclesales_detail as d on v.vehicle_id = d.vehicle_id where date(datetime) = '".$my_date."' GROUP by d.vehicle_id");
            $get_saleman_vehicle->execute();
            $vehicle = $get_saleman_vehicle->get_result();
            $get_saleman_vehicle->close();
            return $vehicle->num_rows;
        }
        
    }

    public function get_jobcardinfo($booking_id){
        $stmt = $this->conn->prepare("select b.bookingdatetime,b.bookingendtime,b.vehicle_id,j.odometerreading,j.notes,j.jobcard_id,b.servicetype_id,v.vin,v.make,v.model,v.year,v.color,a.user_id,a.firstname,a.lastname from tbl_booking as b left join tbl_jobcard as j on b.booking_id = j.booking_id left JOIN tbl_vehicle as v on v.vehicle_id = b.vehicle_id left join tbl_adminuser as a on a.user_id = b.technician_id where b.booking_id = ?");
        $stmt->bind_param("i",$booking_id); 
        $stmt->execute();   
        $jobcard_data = $stmt->get_result();    
        $stmt->close();   
        return $jobcard_data;
    }

    public function displayServiceChecklist($booking_id){
        $data = $this->fetchindividualjob($booking_id);
        // echo "<pre>";print_r($data);exit;

        /* Test code */
        $output = '<table class="service_type_table" style="width:780px;border-collapse:collapse;" border="1">';
        $output .= '<tr>';
				$output .= '<td>';
					
				$output .= '</td>';
        $output .= '</tr>';
        foreach($data['service_provide'] as $providers){
            if($providers['parent_id'] == 0){
                $output .= '<tr>
                        <td colspan="3" class="redtd" style="background-color: #cc0000;color: #fff;padding:13px"><b>'.$providers["service_type_name"].'</b></td>
                </tr>';
            }else{
              $output .='<tr>
                    <td colspan="3"><table style="width:780px;float:left;border-collapse:collapse;">';
                    $output .='<tr><td style="width:30%;padding:5px"><b>'.$providers['service_type_name'].'</b></td>';
                    foreach($providers['service_type_options'] as $options){
                        if($providers['selected_option'] == $options){
                            $color = 'background-color:#ffc000';
                        }else{
                            $color = '';
                        }
                        $output .='<td style="padding:5px;width:20%;'.$color.'">'.$options.'</td>';
                    }   
                    $output .='</tr>
                    </table></td>
                    </tr>';
            }
        }
        $output .= '</table>';
        /* End */
        return $output;
    }

    public function displayJobcardImages($booking_id){
        $data = $this->fetchindividualjob($booking_id);
        $allimages = $data['job_card_images'];
        if(isset($allimages) && $allimages !=''){
            $image_name_array = explode(",",$allimages);
            $output .= '<div class="full-width">';
            foreach($image_name_array as $imgdata){
                $output .= '<span style="margin:0 10px 10px 0;float:left;"><img src="../../imgs/jobcard/images/'.$imgdata.'" width="150" /></span>';
            }
            $output .= "</div>";
        }
        return $output;
    }
    public function updatejobcard_pdf($jobcard_id,$filename){
        $stmt = $this->conn->prepare("update tbl_jobcard set jobcard_pdf = ? where jobcard_id = ?");
        $stmt->bind_param("si",$filename,$jobcard_id);
        if($stmt->execute()){
            return "1";
        }else{
            return "0";
        }

    }

    public function get_site_manager_name($technician_id){
        // Here we get added by user id means sales manager id
        $stmt = $this->conn->prepare("SELECT addedbyuser_id from tbl_adminuser where user_id = ?");
        $stmt->bind_param("i",$technician_id); 
        $stmt->execute();
        $salemanager_data = $stmt->get_result();   
        $salesmanid_data = $salemanager_data->fetch_assoc();
        $stmt->close();
        $salesmanager_id = $salesmanid_data['addedbyuser_id'];
        if(isset($salesmanager_id) && $salesmanager_id != ''){
            $stmt = $this->conn->prepare("SELECT firstname,lastname from tbl_adminuser where user_id = ?");
            $stmt->bind_param("i",$salesmanager_id); 
            $stmt->execute();
            $salemanager_data = $stmt->get_result();   
            $salesmanid_data = $salemanager_data->fetch_assoc();
            $stmt->close();
            $salesmanager_name = $salesmanid_data['firstname']." ".$salesmanid_data['lastname'];
        }
        return $salesmanager_name;
    }

    public function weekly_parts_count_reports($start_date,$end_date,$columname){
        $stmt = $this->conn->prepare("select count(j.".$columname.") as sum_engineoil from tbl_booking as b left join tbl_jobcard as j on b.booking_id = j.booking_id  where b.job_status = 4 AND date(bookingendtime) >= ? AND  date(bookingendtime) <= ?");
        $stmt->bind_param("ss",$start_date,$end_date);        
        $stmt->execute();
        $date_count = $stmt->get_result();
        $my_count = $date_count->fetch_assoc();
        $stmt->close();
        return $my_count['sum_engineoil'];
    }

    public function get_salesmanweekly_report($start_date,$end_date,$salesman_id){
        if(isset($salesman_id) && $salesman_id !=''){
            $get_saleman_vehicle = $this->conn->prepare("select v.vehicle_id from tbl_vehicle as v LEFT join tbl_vehiclesales_detail as d on v.vehicle_id = d.vehicle_id where v.user_id = ?  AND date(datetime) >= '".$start_date."' AND date(datetime) <= '".$end_date."' GROUP by d.vehicle_id");
            $get_saleman_vehicle->bind_param("i",$salesman_id);
            $get_saleman_vehicle->execute();
            $vehicle = $get_saleman_vehicle->get_result();
            $get_saleman_vehicle->close();
            return $vehicle->num_rows;
        }else{
            $get_saleman_vehicle = $this->conn->prepare("select v.vehicle_id from tbl_vehicle as v LEFT join tbl_vehiclesales_detail as d on v.vehicle_id = d.vehicle_id where date(datetime) >= '".$start_date."' AND date(datetime) <= '".$end_date."' GROUP by d.vehicle_id");
            $get_saleman_vehicle->execute();
            $vehicle = $get_saleman_vehicle->get_result();
            $get_saleman_vehicle->close();
            return $vehicle->num_rows;
        }
        
    }

    public function getSalesManRoute($id, $date, $from_time, $to_time, $user_global_id){
		

		if($id == "All"){
			if($from_time != "" && $to_time != ""){
				$from_date = $date." ".$from_time.":00";
				$to_date = $date." ".$to_time.":59";
				$str = " inserttime >= '".$from_date."' AND inserttime <= '".$to_date."'";
			}else{
				$str = "inserttime LIKE '".$date."%'";

			}
			$team_data = array();
			$teams = $this->getSalesMan($user_global_id);
			while ($team = $teams->fetch_assoc()) {
                $tmp = array();
				$tmp['route'] = array();
                $tmp["name"] = $team["firstname"]." ".$team["lastname"];
				$tmp["color"] = $this->rand_color();
				$stmt = $this->conn->prepare("SELECT * FROM  tbl_route WHERE user_id = ? AND ".$str);
				$stmt->bind_param("i",$team['user_id']);
				$result = $stmt->execute();
				$response = $stmt->get_result();
				//$team_data['route'] = array();
				while($row = $response->fetch_assoc()){
					$tmp1 = array();
					$tmp1['lat'] = $row['lat'];
					$tmp1['lan'] = $row['lan'];
					array_push($tmp['route'], $tmp1);
				}
				$stmt->close();
                array_push($team_data, $tmp);
            }
			return $team_data;
		}else{
			$route = array();
			//echo "SELECT * FROM  tbl_route WHERE user_id = 1 AND inserttime LIKE '".$date."%'";
			if($from_time != "" && $to_time != ""){
				$from_date = $date." ".$from_time.":00";
				$to_date = $date." ".$to_time.":59";
				$stmt = $this->conn->prepare("SELECT * FROM  tbl_route WHERE user_id = ? AND inserttime >= '".$from_date."' AND inserttime <= '".$to_date."'");
			}else{
				$stmt = $this->conn->prepare("SELECT * FROM  tbl_route WHERE user_id = ? AND inserttime LIKE '".$date."%'");

			}
			$stmt->bind_param("i",$id);
			$result = $stmt->execute();
			$response = $stmt->get_result();
       
			while($row = $response->fetch_assoc()){
				$tmp = array();
				$tmp['lat'] = $row['lat'];
				$tmp['lan'] = $row['lan'];
				array_push($route, $tmp);

			}
			$stmt->close();
			return $route;
		}
		
		
    }
    
    public function getCenter($id, $date){
		//echo "SELECT * FROM  tbl_route WHERE user_id = 1 AND inserttime LIKE '".$date."%'";
		$stmt = $this->conn->prepare("SELECT AVG(lat) as lat, AVG(lan) as lan FROM  tbl_route WHERE user_id = ? AND inserttime LIKE '".$date."%'");
        $stmt->bind_param("i",$id);
        $result = $stmt->execute();
        $response = $stmt->get_result();
       
		$center = array();
		while($row = $response->fetch_assoc()){
			$tmp = array();
			$tmp['lat'] = $row['lat'];
			$tmp['lan'] = $row['lan'];
			array_push($center, $tmp);

		}
		$stmt->close();
		return $center;
	}

    public function hourlywise_service_completed($start_time,$end_time,$service_id,$pass_start_date,$pass_end_date){
        if(isset($pass_start_date) && isset($pass_end_date)){
            $current_date = $pass_start_date;
            $end_date     = $pass_end_date;
            $stmt = $this->conn->prepare("select count(*) as total_service_completed from tbl_booking where servicetype_id = ? AND job_status = 4 AND date(bookingendtime) BETWEEN ? AND ? AND time(bookingendtime) BETWEEN ? AND ?");
            $stmt->bind_param("issss",$service_id,$pass_start_date,$pass_end_date,$start_time,$end_time);   
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['total_service_completed'];
        }else{
            $current_date = date("Y-m-d");
            $stmt = $this->conn->prepare("select count(*) as total_service_completed from tbl_booking where servicetype_id = ? AND job_status = 4 AND date(bookingendtime) = ? AND time(bookingendtime) BETWEEN ? AND ?");
            $stmt->bind_param("isss",$service_id,$current_date,$start_time,$end_time);   
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['total_service_completed'];
        }
        
    }

    public function hourlywise_parts_completed($start_time,$end_time,$column,$pass_start_date,$pass_end_date){
        if(isset($pass_start_date) && isset($pass_end_date)){
            $current_date = $pass_start_date;
            $end_date     = $pass_end_date;
            $stmt = $this->conn->prepare("select COALESCE(SUM(j.".$column."),0) as sum_engineoil from tbl_booking as b left join tbl_jobcard as j on b.booking_id = j.booking_id  where b.job_status = 4 AND date(bookingendtime) BETWEEN ? AND ? AND time(bookingendtime) BETWEEN ? AND ? ");
            $stmt->bind_param("ssss",$current_date,$end_date,$start_time,$end_time);   
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['sum_engineoil'];
        }else{
            $current_date = date("Y-m-d");
            $stmt = $this->conn->prepare("select COALESCE(SUM(j.".$column."),0) as sum_engineoil from tbl_booking as b left join tbl_jobcard as j on b.booking_id = j.booking_id  where b.job_status = 4 AND date(bookingendtime) = ? AND time(bookingendtime) BETWEEN ? AND ? ");
            $stmt->bind_param("sss",$current_date,$start_time,$end_time);
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['sum_engineoil'];
        }
    }

    public function get_total_time_avg($my_date,$technitions_id){
        // $current_date = date("Y-m-d");
        $stmt = $this->conn->prepare("select COALESCE(avg(total_time),0) as avg_total_time from tbl_booking where job_status = 4 AND technician_id = ? AND date(bookingendtime) = ? ");
        $stmt->bind_param("is",$technitions_id,$my_date);
        $stmt->execute();
        $date_count = $stmt->get_result();
        $my_count = $date_count->fetch_assoc();
        $stmt->close();
        return $my_count['avg_total_time'];
    }

    public function get_technition_productivity_daily($my_date,$service_id,$technician_id){
        $stmt = $this->conn->prepare("select COALESCE(avg(total_time),0) as avg_total_time from tbl_booking where job_status = 4 AND technician_id = $technician_id AND date(bookingendtime) = '".$my_date."' AND servicetype_id = $service_id");
        $stmt->execute();
        $date_count = $stmt->get_result();
        $my_count = $date_count->fetch_assoc();
        $stmt->close();
        return $my_count['avg_total_time'];
    }

    public function get_technitions_report_hourly($start_time,$end_time,$technician_id,$pass_start_date,$pass_end_date){
        if(isset($pass_start_date) && isset($pass_end_date)){
            $current_date = $pass_start_date;
            $end_date     = $pass_end_date;
            if(isset($technician_id) && $technician_id != ''){
                $stmt = $this->conn->prepare("select count(*) as total_service_completed from tbl_booking  where job_status = 4 AND technician_id = ? AND date(bookingendtime) BETWEEN ? AND ? AND time(bookingendtime) BETWEEN ? AND ?");
                $stmt->bind_param("issss",$technician_id,$current_date,$end_date,$start_time,$end_time);         
            }else{
                $stmt = $this->conn->prepare("select count(*) as total_service_completed from tbl_booking  where job_status = 4 AND date(bookingendtime) BETWEEN ? AND ? AND time(bookingendtime) BETWEEN ? AND ?");
                $stmt->bind_param("ssss",$current_date,$end_date,$start_time,$end_time);         
            }
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            
            return $my_count['total_service_completed'];

        }else{
            
            $current_date = date("Y-m-d");
            if(isset($technician_id) && $technician_id != ''){
                $stmt = $this->conn->prepare("select count(*) as total_service_completed from tbl_booking where job_status = 4 AND date(bookingendtime) = '".$current_date."' AND technician_id = $technician_id AND time(bookingendtime) BETWEEN '$start_time' AND '$end_time'");
            }else{
                
                $stmt = $this->conn->prepare("select count(*) as total_service_completed from tbl_booking where job_status = 4 AND date(bookingendtime) = ? AND time(bookingendtime) BETWEEN ? AND ? ");
                $stmt->bind_param("sss",$current_date,$start_time,$end_time);                                         
            }
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['total_service_completed'];

        }
    }

    public function get_houveryly_time_avg($start_time,$end_time,$technitions_id,$pass_start_date,$pass_end_date){
        if(isset($pass_start_date) && isset($pass_end_date)){
            $current_date = $pass_start_date;
            $end_date     = $pass_end_date;
            $stmt = $this->conn->prepare("select COALESCE(avg(total_time),0) as avg_total_time from tbl_booking  where job_status = 4 AND technician_id = ? AND date(bookingendtime) BETWEEN ? AND ? AND time(bookingendtime) BETWEEN ? AND ?");
            $stmt->bind_param("issss",$technitions_id,$current_date,$end_date,$start_time,$end_time);         
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['avg_total_time'];
        }else{
            $current_date = date("Y-m-d");
            $stmt = $this->conn->prepare("select COALESCE(avg(total_time),0) as avg_total_time  from tbl_booking  where job_status = 4 AND technician_id = ? AND date(bookingendtime) = ? AND time(bookingendtime) BETWEEN ? AND ?");
            $stmt->bind_param("isss",$technitions_id,$current_date,$start_time,$end_time);         
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['avg_total_time'];

        }
    }

    public function sendAttachmentmail($useremailid,$reportname,$filename,$sub,$content,$foldername){
        $to = $useremailid;
        
        // $to = "mohit@opal.in";
        $from = 'info@kwmapp.com';
        $fromName = 'KWM App';
        $subject = $sub;
        $file = '../../imgs/reports/'.$foldername.'/'.$filename;
        $htmlContent = $content;
        $headers = "From: $fromName"." <".$from.">";
        $semi_rand = md5(time()); 
        $mime_boundary = "==Multipart_Boundary_x{$semi_rand}x"; 
        $headers .= "\nMIME-Version: 1.0\n" . "Content-Type: multipart/mixed;\n" . " boundary=\"{$mime_boundary}\""; 
        $message = "--{$mime_boundary}\n" . "Content-Type: text/html; charset=\"UTF-8\"\n" .
        "Content-Transfer-Encoding: 7bit\n\n" . $htmlContent . "\n\n"; 
        //preparing attachment
        if(!empty($file) > 0){
            if(is_file($file)){
                $message .= "--{$mime_boundary}\n";
                $fp =    @fopen($file,"rb");
                $data =  @fread($fp,filesize($file));
                @fclose($fp);
                $data = chunk_split(base64_encode($data));
                $message .= "Content-Type: application/octet-stream; name=\"".basename($file)."\"\n" . 
                "Content-Description: ".basename($file)."\n" .
                "Content-Disposition: attachment;\n" . " filename=\"".basename($file)."\"; size=".filesize($file).";\n" . 
                "Content-Transfer-Encoding: base64\n\n" . $data . "\n\n";
            }
        }
        $message .= "--{$mime_boundary}--";
        $returnpath = "-f" . $from;
        //send email
        if(@mail($to, $subject, $message, $headers, $returnpath)){
            return 1;
        }else{
            return 0;
        } 
    }

    public function get_technician_houveryly_time_avg($start_time,$end_time,$technician_id,$pass_start_date,$pass_end_date,$service_id){
        if(isset($pass_start_date) && isset($pass_end_date)){
            $current_date = $pass_start_date;
            $end_date     = $pass_end_date;
            $stmt = $this->conn->prepare("select COALESCE(avg(total_time),0) as avg_total_time  from tbl_booking  where job_status = 4 AND technician_id = ? AND servicetype_id = ? AND date(bookingendtime) BETWEEN ? AND ? AND time(bookingendtime) BETWEEN ? AND ?");
            $stmt->bind_param("iissss",$technician_id,$service_id,$current_date,$end_date,$start_time,$end_time);         
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['avg_total_time'];            
        }else{
            $current_date = date("Y-m-d");
            $stmt = $this->conn->prepare("select COALESCE(avg(total_time),0) as avg_total_time  from tbl_booking  where job_status = 4 AND technician_id = ? AND servicetype_id = ? AND date(bookingendtime) = ? AND time(bookingendtime) BETWEEN ? AND ?");
            $stmt->bind_param("iisss",$technician_id,$service_id,$current_date,$start_time,$end_time);         
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['avg_total_time'];

        }
    }

    public function get_daily_technician_productivity($my_date,$technician_id){
        $stmt = $this->conn->prepare("SELECT DATE_FORMAT(b.bookingdatetime,'%H:%i:%s') as starttime ,DATE_FORMAT(b.bookingendtime,'%H:%i:%s') as finishtime ,b.duration as timetaken,s.name as servicename,u.firstname,u.lastname FROM `tbl_booking` as b left join tbl_servicetype as s on s.servicetype_id = b.servicetype_id LEFT JOIN tbl_adminuser as u on u.user_id = b.technician_id where b.job_status = 4 AND b.technician_id = $technician_id AND date(b.bookingdatetime) = '$my_date' AND date(b.bookingendtime) = '$my_date'");
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }

    public function get_daily_site_productivity($my_date,$technitions_comma_seprated){
        $stmt = $this->conn->prepare("SELECT DATE_FORMAT(b.bookingdatetime,'%H:%i:%s') as starttime ,DATE_FORMAT(b.bookingendtime,'%H:%i:%s') as finishtime ,b.duration as timetaken,s.name as servicename,u.firstname,u.lastname FROM `tbl_booking` as b left join tbl_servicetype as s on s.servicetype_id = b.servicetype_id LEFT JOIN tbl_adminuser as u on u.user_id = b.technician_id where b.job_status = 4 AND b.technician_id IN ($technitions_comma_seprated) AND date(b.bookingdatetime) = '$my_date' AND date(b.bookingendtime) = '$my_date'");
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }

    public function get_hourly_site_productivity($start_time,$end_time,$pass_start_date, $pass_end_date,$technitions_comma_seprated){
        if($pass_start_date != '' && $pass_end_date != ''){
            // echo "SELECT DATE_FORMAT(b.bookingdatetime,'%H:%i:%s') as starttime ,DATE_FORMAT(b.bookingendtime,'%H:%i:%s') as finishtime ,b.duration as timetaken,s.name as servicename,u.firstname,u.lastname FROM `tbl_booking` as b left join tbl_servicetype as s on s.servicetype_id = b.servicetype_id LEFT JOIN tbl_adminuser as u on u.user_id = b.technician_id where b.job_status = 4 AND b.technician_id IN ($technitions_comma_seprated) AND date(b.bookingendtime) BETWEEN '$pass_start_date' AND '$pass_end_date' AND time(b.bookingendtime) BETWEEN '$start_time' AND '$end_time'"."\n\n";
            $stmt = $this->conn->prepare("SELECT DATE_FORMAT(b.bookingdatetime,'%H:%i:%s') as starttime ,DATE_FORMAT(b.bookingendtime,'%H:%i:%s') as finishtime ,b.duration as timetaken,s.name as servicename,u.firstname,u.lastname FROM `tbl_booking` as b left join tbl_servicetype as s on s.servicetype_id = b.servicetype_id LEFT JOIN tbl_adminuser as u on u.user_id = b.technician_id where b.job_status = 4 AND b.technician_id IN ($technitions_comma_seprated) AND date(b.bookingendtime) BETWEEN '$pass_start_date' AND '$pass_end_date' AND time(b.bookingendtime) BETWEEN '$start_time' AND '$end_time'");
            $stmt->execute();
            $data = $stmt->get_result();
            $stmt->close();
            return $data;
        }else{
            $current_date = date("Y-m-d");
            $stmt = $this->conn->prepare("SELECT DATE_FORMAT(b.bookingdatetime,'%H:%i:%s') as starttime ,DATE_FORMAT(b.bookingendtime,'%H:%i:%s') as finishtime ,b.duration as timetaken,s.name as servicename,u.firstname,u.lastname FROM `tbl_booking` as b left join tbl_servicetype as s on s.servicetype_id = b.servicetype_id LEFT JOIN tbl_adminuser as u on u.user_id = b.technician_id where b.job_status = 4 AND b.technician_id IN ($technitions_comma_seprated) AND date(b.bookingdatetime) = '$current_date' AND date(b.bookingendtime) = '$current_date' AND time(b.bookingendtime) BETWEEN '$start_time' AND '$end_time'");
            $stmt->execute();
            $data = $stmt->get_result();
            $stmt->close();
            return $data;
        }
    }

    public function get_weekly_site_productivity($start_date, $end_date,$technitions_comma_seprated){
        $stmt = $this->conn->prepare("SELECT DATE_FORMAT(b.bookingdatetime,'%H:%i:%s') as starttime ,DATE_FORMAT(b.bookingendtime,'%H:%i:%s') as finishtime ,b.duration as timetaken,s.name as servicename,u.firstname,u.lastname FROM `tbl_booking` as b left join tbl_servicetype as s on s.servicetype_id = b.servicetype_id LEFT JOIN tbl_adminuser as u on u.user_id = b.technician_id where b.job_status = 4 AND b.technician_id IN ($technitions_comma_seprated) AND date(b.bookingendtime) >= '$start_date' AND date(b.bookingendtime) <= '$end_date'");
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }

    public function get_weekly_technician_productivity($start_date, $end_date,$technician_id){
        $stmt = $this->conn->prepare("SELECT DATE_FORMAT(b.bookingdatetime,'%H:%i:%s') as starttime ,DATE_FORMAT(b.bookingendtime,'%H:%i:%s') as finishtime ,b.duration as timetaken,s.name as servicename,u.firstname,u.lastname FROM `tbl_booking` as b left join tbl_servicetype as s on s.servicetype_id = b.servicetype_id LEFT JOIN tbl_adminuser as u on u.user_id = b.technician_id where b.job_status = 4 AND b.technician_id = $technician_id AND date(b.bookingendtime) >= '$start_date' AND date(b.bookingendtime) <= '$end_date'");
        $stmt->execute();
        $data = $stmt->get_result();
        $stmt->close();
        return $data;
    }

    public function get_technitions_daily_completed_services($my_date,$technician_id){
        if(isset($technician_id) && $technician_id != ''){
            $stmt = $this->conn->prepare("select count(*) as total_service_completed from tbl_booking where job_status = 4 AND date(bookingendtime) = ? AND technician_id = ?");
            $stmt->bind_param("ss",$my_date,$technician_id);
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['total_service_completed'];
        }else{
            $stmt = $this->conn->prepare("select count(*) as total_service_completed from tbl_booking where job_status = 4 AND date(bookingendtime) = ?");
            $stmt->bind_param("s",$my_date);
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['total_service_completed'];

        }
        

    }

    public function hourlywise_parts_count_completed($start_time,$end_time,$column,$pass_start_date,$pass_end_date){
        if(isset($pass_start_date) && isset($pass_end_date)){
            $current_date = $pass_start_date;
            $end_date     = $pass_end_date;
            $stmt = $this->conn->prepare("select count(j.".$column.") as sum_engineoil from tbl_booking as b left join tbl_jobcard as j on b.booking_id = j.booking_id  where b.job_status = 4 AND date(bookingendtime) BETWEEN ? AND ? AND time(bookingendtime) BETWEEN ? AND ?");
            $stmt->bind_param("ssss",$current_date,$end_date,$start_time,$end_time);   
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['sum_engineoil']; 
        }else{
            $current_date = date("Y-m-d");
            $stmt = $this->conn->prepare("select count(j.".$column.") as sum_engineoil from tbl_booking as b left join tbl_jobcard as j on b.booking_id = j.booking_id  where b.job_status = 4 AND date(bookingendtime) = ? AND time(bookingendtime) BETWEEN ? AND ? ");
            $stmt->bind_param("sss",$current_date,$start_time,$end_time);
            $stmt->execute();
            $date_count = $stmt->get_result();
            $my_count = $date_count->fetch_assoc();
            $stmt->close();
            return $my_count['sum_engineoil'];
        }
    }



    public function get_service_type(){
       $stmt = $this->conn->prepare("select * from tbl_servicetype");
       $stmt->execute();
       $allservices = $stmt->get_result();
       $stmt->close();
       return $allservices;
    }

    public function get_conversation($conversation_id){
        $stmt = $this->conn->prepare("select m.*,u.firstname as sendername,a.firstname as receivername,u.profilepic from tbl_message as m left join tbl_adminuser as u on u.user_id = m.sender_id left join tbl_adminuser as a on a.user_id = m.receiver_id where conversation_id = ?");
        $stmt->bind_param("i",$conversation_id);
        $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }

    public function get_enquiry_conversation($inquiryid){
        $stmt = $this->conn->prepare("select iq.* from tbl_inquirydetails as iq left join tbl_inquiry as i on i.inquiry_id = iq.inquiry_id where iq.inquiry_id = ?");
        $stmt->bind_param("i",$inquiryid);
        $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }

    public function update_inquiry_status($inquiryid,$filename){
        $status = 1;
        $stmt = $this->conn->prepare("update tbl_inquiry set file_status = ? , pdf_filename = ? where inquiry_id = ?");
        $stmt->bind_param("isi",$status,$filename,$inquiryid);
        if($stmt->execute()){
            return 1;
        }else{
            return 0;
        }

    }

    

    public function get_customer_name($receiverid){
        
        $stmt = $this->conn->prepare("select firstname from tbl_customer where user_id = ?");
        $stmt->bind_param("i",$receiverid);
        $stmt->execute();
        $response = $stmt->get_result();
        $receiver_name_data = $response->fetch_assoc();
        
        return $receiver_name_data['firstname'];
    }

    public function get_image_sitemanager($sender_id,$imageof){

        if($imageof == 0){ // site manager
            $stmt = $this->conn->prepare("select profilepic from tbl_adminuser where user_id = ?");
            $stmt->bind_param("i",$sender_id);
            $stmt->execute();
            $response = $stmt->get_result();
            $receiver_name_data = $response->fetch_assoc();
            return $receiver_name_data['profilepic'];
        }else{
            $stmt = $this->conn->prepare("select profilepic from tbl_customer where user_id = ?");
            $stmt->bind_param("i",$sender_id);
            $stmt->execute();
            $response = $stmt->get_result();
            $receiver_name_data = $response->fetch_assoc();
            return $receiver_name_data['profilepic'];
        }

        
    }

    



    public function update_status_pdfname($conversation_id,$filename){
        $status = 1;
        $stmt = $this->conn->prepare("UPDATE tbl_conversation SET pdf_filename = ? , status = ? where conversation_id = ?");
        $stmt->bind_param("sii",$filename,$status,$conversation_id);
        if($stmt->execute()){
            return 1;
        }else{
            return 0;
        }
    }

    function getSitemanagerByCustomer($user_id){
		
		$stmt = $this->conn->prepare("SELECT addedbyuser_id FROM `tbl_customer` WHERE user_id = ?");
        $stmt->bind_param("i", $user_id);
        $result = $stmt->execute();

        if (false === $result) {
            die('execute() failed: ' . htmlspecialchars($stmt->error));
        }
		$customer_data_res = $stmt->get_result();
		$customer_data = $customer_data_res->fetch_assoc();
        $stmt->close();
		//print_r($customer_data);die();
		$stmt = $this->conn->prepare("SELECT addedbyuser_id,usertype_id FROM `tbl_adminuser` WHERE user_id = ?");
        $stmt->bind_param("i", $customer_data['addedbyuser_id']);
        $result = $stmt->execute();

        if (false === $result) {
            die('execute() failed: ' . htmlspecialchars($stmt->error));
        }
		$customer_sales_man_res = $stmt->get_result();
        $stmt->close();
		$customer_sales_man = $customer_sales_man_res->fetch_assoc();
		if($customer_sales_man['usertype_id'] == "1"){

			$stmt = $this->conn->prepare("SELECT addedbyuser_id,usertype_id FROM `tbl_adminuser` WHERE user_id = ?");
			$stmt->bind_param("i", $customer_sales_man['addedbyuser_id']);
			$result = $stmt->execute();
			if (false === $result) {
				die('execute() failed: ' . htmlspecialchars($stmt->error));
			}
			$customer_sales_manager_res = $stmt->get_result();
			$customer_sales_manager = $customer_sales_manager_res->fetch_assoc();
			$stmt->close();
			$sitemanager_id = $customer_sales_manager['addedbyuser_id'];
		}else{
			$sitemanager_id = $customer_sales_man['addedbyuser_id'];
		}
		return $sitemanager_id;
    }
    public function getAdminUser($admin_id){
		$stmt = $this->conn->prepare("SELECT * FROM `tbl_adminuser` WHERE user_id = ?");
		$stmt->bind_param("i", $admin_id);
		$result = $stmt->execute();
		if (false === $result) {
			die('execute() failed: ' . htmlspecialchars($stmt->error));
		}
		$customer_sales_manager_res = $stmt->get_result();
		$customer_sales_manager = $customer_sales_manager_res->fetch_assoc();
		return $customer_sales_manager;
    }
    
    public function getCustomer($user_id) {
        $stmt = $this->conn->prepare("SELECT * FROM `tbl_customer` WHERE user_id = ?");
        $stmt->bind_param("i", $user_id);
        $result = $stmt->execute();

        if (false === $result) {
            die('execute() failed: ' . htmlspecialchars($stmt->error));
        }
		$response = $stmt->get_result();
        $stmt->close();
        return $response;
    }

    public function getUserFeedback($conversation_id){
        
        $stmt = $this->conn->prepare("SELECT * FROM `tbl_message` WHERE conversation_id = ? ORDER BY conversation_datetime ASC");
        $stmt->bind_param("i", $conversation_id);
        $result = $stmt->execute();
        if (false === $result) {
            die('execute() failed: ' . htmlspecialchars($stmt->error));
        }
        $conversation = $stmt->get_result();
        $stmt->close();
        return $conversation;
	}
    
    


    public function sent_message($conversation_id,$message,$sender_id,$receiver_id,$send_by,$conversation_datetime){
        $stmt = $this->conn->prepare("INSERT INTO tbl_message (conversation_id,sender_id,receiver_id,message,send_by,conversation_datetime) VALUES (?,?,?,?,?,?)");
        $stmt->bind_param("iiisis",$conversation_id,$sender_id,$receiver_id,$message,$send_by,$conversation_datetime); 
        if($stmt->execute()){
            $stmt->close();
            return 1;
        }else{
            return 2;   
        }
    }

    public function chk_record_exist($table,$field,$email,$technician_id){
        if(isset($technician_id) && $technician_id !=''){
            $check_query = $this->conn->prepare("SELECT user_id FROM tbl_adminuser WHERE emailaddress = ? AND user_id != ?");
            $check_query->bind_param("si",$email,$technician_id);
            $check_query->execute();
            $user_data = $check_query->get_result();

        }else{
            $check_query = $this->conn->prepare("SELECT user_id FROM tbl_adminuser WHERE emailaddress = ?");
            $check_query->bind_param("s",$email);
            $check_query->execute();
            $user_data = $check_query->get_result();
        }
        
        if($user_data->num_rows > 0){
            return 1;
        }else{
            return 0;
        }
    }

	public function getAdmindata($admin_id){
        // SELECT accountname, accountnumber, bankname, emailaddress, mobilenumber, address, address2 FROM `tbl_adminuser` as a, tbl_bankaccountdetail as b WHERE a.user_id = 4 AND b.user_id=a.user_id AND b.user_type = 3
		$stmt = $this->conn->prepare("SELECT b.accountname, b.accountnumber, b.bankname, a.emailaddress, a.mobilenumber, a.address, a.address2 FROM `tbl_adminuser` as a left join tbl_bankaccountdetail as b on a.user_id = b.user_id where a.user_id = ?");
		$stmt->bind_param("i", $admin_id);
		$result = $stmt->execute();
		if (false === $result) {
			die('execute() failed: ' . htmlspecialchars($stmt->error));
		}
		$admin_data_res = $stmt->get_result();
		$admin_data = $admin_data_res->fetch_assoc();
		return $admin_data;
	}
	function updateAdmin($user_id, $addressLine1, $addressLine2, $mobile, $email, $accountname, $accountnumber, $bankname){
		//$result = $this->getCustomer($user_id);
		//$data = $result->fetch_assoc();
		$user_type = 1;
		//print_r($data);
		//die();
		$stmt = $this->conn->prepare("INSERT INTO tbl_contactdetails (user_id, address1, address2, mobile, email, user_type) VALUES (?, ?, ?, ?, ?, ?) ");
		$stmt->bind_param("issssi", $user_id, $addressLine1, $addressLine2, $mobile, $email, $user_type);
        $stmt->execute();
		$stmt->close();
		
		
		$stmt = $this->conn->prepare("UPDATE  tbl_adminuser set address = ?, address2 = ?, emailaddress = ?, mobilenumber= ? WHERE user_id = ?");
        $stmt->bind_param("ssssi", $addressLine1, $addressLine2, $email, $mobile, $user_id);
        $stmt->execute();
        $num_affected_rows = $stmt->affected_rows;
        $stmt->close();
		
		$stmt = $this->conn->prepare("UPDATE  tbl_bankaccountdetail set accountname = ?, accountnumber = ?, bankname = ? WHERE user_id = ?");
        $stmt->bind_param("sssi", $accountname, $accountnumber, $bankname, $user_id);
        $stmt->execute();
        $num_affected_rows = $stmt->affected_rows;
        $stmt->close();
        return $num_affected_rows > 0;
	}

    public function fetchAllpartsorder($user_id){
        $dateTo =  date("Y-m-d")." 00:00:00";
        $dateFrom = date("Y-m-d",strtotime(date("Y-m-d", strtotime($dateTo))." + 1 day"))." 23:59:59";
        $stmt = $this->conn->prepare("SELECT a.*,c.* FROM tbl_job a left join tbl_booking b on b.booking_id = a.booking_id join tbl_vehicle c on c.vehicle_id = b.vehicle_id WHERE a.technician_id = ? AND a.status NOT IN(3,4) AND a.scheduledatetime BETWEEN ? AND ? ");
        $stmt->bind_param("iss",$user_id,$dateTo,$dateFrom);
        $result = $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }
    public function fetchindividualjob($booking_id){
        $myarray['service_parts'] = array();
        $myarray['service_provide'] = array();
        $todays_date = date("Y-m-d");
        // Here we fetch the vehicle id of booking  
            $allbooing_data = $this->conn->prepare("SELECT b.vehicle_id, b.servicetype_id,a.firstname,jc.jobcard_id, odometerreading, notes, oilfilter_number, airfilter_number, engineoil_amount, breakfluid_amount, clutchfluid_amount, coolant_amount, job_card_images FROM tbl_booking b, tbl_jobcard jc,tbl_adminuser as a WHERE  b.booking_id = jc.booking_id and a.user_id = b.technician_id and  b.booking_id = ? "); 
            $allbooing_data->bind_param("i",$booking_id);
            $result_vehicledata = $allbooing_data->execute();
            $vehil_result = $allbooing_data->get_result();
            $vehicledata  = $vehil_result->fetch_assoc();
            $vehicle_id = $vehicledata['vehicle_id'];
			$servicetype_id = $vehicledata['servicetype_id'];
			$jobcard_id = $vehicledata['jobcard_id'];
			$myarray["odometerreading"] = $vehicledata['odometerreading'];
            $myarray["notes"] = $vehicledata['notes'];
            $myarray["technician_name"] = $vehicledata['firstname'];
			$myarray["oilfilter_number"] = $vehicledata['oilfilter_number'];
			$myarray["engineoil_amount"] = $vehicledata['engineoil_amount'];
			$myarray["breakfluid_amount"] = $vehicledata['breakfluid_amount'];
			$myarray["clutchfluid_amount"] = $vehicledata['clutchfluid_amount'];
			$myarray["coolant_amount"] = $vehicledata['coolant_amount'];
			$myarray["job_card_images"] = $vehicledata['job_card_images'];
			$myarray["airfilter_number"] = $vehicledata['airfilter_number'];
			$myarray["vehicle_id"] = $vehicledata['vehicle_id'];
            // here we fetch service type name 
            $service_type_query =  $this->conn->prepare("SELECT name FROM tbl_servicetype WHERE servicetype_id = ? "); 
            $service_type_query->bind_param("i",$servicetype_id);
            $service_type_query->execute();
            $service_type_query_result = $service_type_query->get_result();
            $service_type_query_data  = $service_type_query_result->fetch_assoc();
            $service_name = $service_type_query_data['name'];
        // End 
            
        // accoding to service id we fetch all service 
            $allservices_query = $this->conn->prepare("SELECT * FROM tbl_jobcard_detail WHERE jobcard_id = ? "); 
            $allservices_query->bind_param("i",$jobcard_id);
            $allservices_query->execute();
            $allservices_query_result = $allservices_query->get_result();
            while($allservices_query_data = $allservices_query_result->fetch_assoc()){
                //print_r($allservices_query_data);
				$tmp = array();
				$tmp['jobcard_detail_id'] = $allservices_query_data['jobcard_detail_id'];
				$tmp['jobcard_id'] = $allservices_query_data['jobcard_id'];
				$tmp['service_type_name'] = $allservices_query_data['service_type_name'];
				$tmp['service_type_options'] = explode(",",$allservices_query_data['service_type_options']);
				$tmp['parent_id'] = $allservices_query_data['parent_id'];
				$tmp['selected_option'] = $allservices_query_data['selected_option'];
				 array_push($myarray['service_provide'],$tmp);
            }
               
            
        // here we fetch the current vehicle service vehicle details 
            $vehiclinfo_query = $this->conn->prepare("SELECT * FROM tbl_vehicle WHERE vehicle_id = ? "); 
            $vehiclinfo_query->bind_param("i",$vehicle_id);
            $vehiclinfo_query->execute();
            $vehicleinfo_result = $vehiclinfo_query->get_result();
            $vehicleinfo = $vehicleinfo_result->fetch_assoc();
            $myarray["plat_no"] = $vehicleinfo['plateno'];
            $myarray["model"] = $vehicleinfo['model'];
            $myarray["service_type"] = $service_name;
			$myarray["service_type_id"] = $servicetype_id;
		
        // End 
        return $myarray;
        // End
    }
    // Here we fetch the contract hoistory
    public function fetchcontracthistory($user_id){
        $allcontact = $this->conn->prepare("SELECT * FROM tbl_adminuser WHERE addedbyuser_id = ? ");
        $allcontact->bind_param("i", $user_id);
        $allcontact->execute();
        $allcontact_data = $allcontact->get_result();
        while($data = $allcontact_data->fetch_assoc()){
            $all_contactor[] = $data;
        }
        return $all_contactor;
    }
    // End 

    public function get_customer_all_inquiry($user_id){
        $stmt = $this->conn->prepare("select * from tbl_inquiry where sender_id = ?");
        $stmt->bind_param("i",$user_id);
        $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }

    public function get_all_inquiry(){
        $stmt = $this->conn->prepare("select i.*,c.firstname from tbl_customer as c right join tbl_inquiry as i on c.user_id = i.sender_id where i.file_status = 0");
        $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;
    }

    public function get_all_conversation($customer_id){
        $stmt = $this->conn->prepare("SELECT * FROM `tbl_conversation` where sender_id = ? OR receiver_id = ?");
        $stmt->bind_param('ii',$customer_id,$customer_id);
        $stmt->execute();
        $response = $stmt->get_result();
        $stmt->close();
        return $response;

    }

    public function updateBookingStatus($notes,$booking_id){
        $stmt = $this->conn->prepare("UPDATE tbl_booking SET job_status = '4' WHERE booking_id = ? ");
        $stmt->bind_param("i",$booking_id);
        $stmt->execute();
        $stmt->close();
        

        $stmt1 = $this->conn->prepare("UPDATE tbl_jobcard SET notes = ? WHERE booking_id = ? ");
        $stmt1->bind_param("si",$notes,$booking_id);
        if($stmt1->execute()){
            $stmt1->close();
            return 1;
        }else{
            $stmt1->close();
            return 0;
        }
    }
    
    // Here  we  create the function to update details
    public function updateCustomer($user_id, $addressLine1, $addressLine2, $mobile, $email){
        $result = $this->getCustomer($user_id);
        $data = $result->fetch_assoc();
        $user_type = 0;
        $stmt = $this->conn->prepare("INSERT INTO tbl_contactdetails (user_id, address1, address2, mobile, email, user_type) VALUES (?, ?, ?, ?, ?, ?) ");
        $stmt->bind_param("issssi", $user_id, $addressLine1, $addressLine2, $mobile, $email, $user_type);
        $stmt->execute();
        $stmt->close();

	    $stmt = $this->conn->prepare("UPDATE  tbl_customer set address = ?, address2 = ?, emailaddress = ?, mobilenumber= ? WHERE user_id = ?");
        $stmt->bind_param("ssssi", $addressLine1, $addressLine2, $email, $mobile, $user_id);
        $stmt->execute();
        $num_affected_rows = $stmt->affected_rows;
        $stmt->close();
        return $num_affected_rows > 0;
    }
    // End
    
    
    public function isValidApiKey($api_key) {
        $stmt = $this->conn->prepare("SELECT emailaddress from tbl_adminuser WHERE api_key = ?");
        $stmt->bind_param("s", $api_key);
        $stmt->execute();
        $stmt->store_result();
        $num_rows = $stmt->num_rows;
        //echo $num_rows;
        $stmt->close();
        return $num_rows > 0;
    }
    
}